package ru.yandex.direct.uaas;


import java.util.Arrays;
import java.util.Base64;
import java.util.List;

import static java.util.stream.Collectors.toUnmodifiableList;

public class UaasResponse {

    private final String configVersion;
    private final String splitParams;
    private final String boxes;
    private final String boxedCrypted;
    private final List<String> flags;

    private UaasResponse(
            String configVersion,
            String splitParams,
            String boxes,
            String boxedCrypted,
            List<String> flags
    ) {
        this.splitParams = splitParams;
        this.configVersion = configVersion;
        this.boxes = boxes;
        this.boxedCrypted = boxedCrypted;
        this.flags = flags;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getConfigVersion() {
        return configVersion;
    }

    public String getSplitParams() {
        return splitParams;
    }

    public String getBoxes() {
        return boxes;
    }

    public String getBoxedCrypted() {
        return boxedCrypted;
    }

    public List<String> getFlags() {
        return flags;
    }

    public static class Builder {
        private String configVersion;
        private String splitParamsEncoded;
        private String boxes;
        private String boxedCrypted;
        private String flagsEncoded;

        public Builder withConfigVersion(String configVersion) {
            this.configVersion = configVersion;
            return this;
        }

        public Builder withSplitParamsEncoded(String splitParamsEncoded) {
            this.splitParamsEncoded = splitParamsEncoded;
            return this;
        }

        public Builder withBoxes(String boxes) {
            this.boxes = boxes;
            return this;
        }

        public Builder withBoxedCrypted(String boxedCrypted) {
            this.boxedCrypted = boxedCrypted;
            return this;
        }

        public Builder withFlagsEncoded(String flagsEncoded) {
            this.flagsEncoded = flagsEncoded;
            return this;
        }

        public UaasResponse build() {
            var splitParams = "".equals(splitParamsEncoded) ? "" : decodeBase64(splitParamsEncoded);
            List<String> flags = "".equals(flagsEncoded) ? List.of() :
                    Arrays.stream(flagsEncoded.split(",")).map(this::decodeBase64).collect(toUnmodifiableList());
            return new UaasResponse(
                    configVersion,
                    splitParams,
                    boxes,
                    boxedCrypted,
                    flags
            );
        }

        private String decodeBase64(String encode) {
            return new String(Base64.getDecoder().decode(encode));
        }
    }

    @Override
    public String toString() {
        return "UaasResponse{" +
                "configVersion='" + configVersion + '\'' +
                ", splitParams='" + splitParams + '\'' +
                ", boxes='" + boxes + '\'' +
                ", boxedCrypted='" + boxedCrypted + '\'' +
                ", flags=" + flags +
                '}';
    }
}
