package ru.yandex.direct.ugcdb.client;

import java.util.List;

import javax.annotation.Nullable;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.kernel.ugc.protos.direct.TDirectReview;
import ru.yandex.kernel.ugc.protos.direct.TDirectService;

import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;

/**
 * Клиент для работы с внутренним HTTP API для доступа к UGC DB
 * <p>
 * UGC DB -- в зоне ответственности UGC. Вики: https://wiki.yandex-team.ru/JandeksPoisk/UGC/
 * Чат поддержки: ugc-support https://t.me/joinchat/Bcdr-gzGQCILjbW7rXPZ9g
 */
public class UgcDbClient {
    Api api;

    public UgcDbClient(String url, TvmService tvmService, ParallelFetcherFactory parallelFetcherFactory,
                       TvmIntegration tvmIntegration) {
        api = createApi(url, tvmService, parallelFetcherFactory, tvmIntegration);
    }

    Api createApi(String url, TvmService tvmService, ParallelFetcherFactory parallelFetcherFactory,
                  TvmIntegration tvmIntegration) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withResponseConverterFactory(
                        ResponseConverterFactory.builder()
                                .addConverters(
                                        new TDirectReviewConverter(),
                                        new TDirectServiceConverter()
                                )
                                .build()
                )
                .useTvm(tvmIntegration, tvmService)
                .withProfileName("ugc_db_client")
                .withBaseUrl(url)
                .build()
                .create(Api.class);
    }


    /**
     * Получение списка отзывов на фрилансера
     */
    public List<TDirectReview> getFeedbackList(Long freelancerId) {
        Result<TDirectService> result =
                api.getFeedbackList(freelancerId, "ORDER BY $.CreateTime/seconds DESC").execute();
        checkResultForErrors(result, UgcDbClientException::new);
        return result.getSuccess().getReviewsList();
    }

    /**
     * Получение отзыва
     *
     * @return если отзыв не найден, то будет null
     */
    @Nullable
    public TDirectReview getFeedback(Long freelancerId, String feedbackId) {
        Result<TDirectReview> result = api.getFeedback(freelancerId, feedbackId).execute();
        checkResultForErrors(result, UgcDbClientException::new);
        return result.getSuccess();
    }

    /**
     * Добавление или обновление записи отзыва
     */
    public void saveFeedback(Long freelancerId, String feedbackId, TDirectReview review) throws UgcDbClientException {
        Result<Void> result = api.saveFeedback(freelancerId, feedbackId, new UgcDbRequest(review)).execute();
        checkResultForErrors(result, UgcDbClientException::new);
    }

    /**
     * Удаление отзыва
     */
    public void deleteFeedback(Long freelancerId, String feedbackId) throws UgcDbClientException {
        Result<Void> result = api.deleteFeedback(freelancerId, feedbackId).execute();
        checkResultForErrors(result, UgcDbClientException::new);
    }
}
