package ru.yandex.direct.utils;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * Асинхронный аналог стандартного Consumer\<T\>.
 * <p>
 * Объект, переданный в accept складируется в BlockingChannel, отдельный тред выгребает из канала элементы
 * и обрабатывает их с помощью указанного в конструкторе Consumer\<T\>.
 * <p>
 * Простейший пример использования:
 * <p>
 * try (AsyncConsumer\<Object\> consumer = new AsyncConsumer(item -> System.out.println(item.toString()), 10)) {
 * consumer.accept(1);
 * consumer.accept(2);
 * }
 * <p>
 * Если канал переполнен, accept блокируется.
 * Если обработка элемента завершилась с ошибкой, то канал закрывается, после чего accept будет возвращать исключение.
 * Элемент, на котором случилась ошибка и все последующие сохраняются в специальном списке пофейленых, их можно получить
 * через getFailedItems.
 * <p>
 * AsyncConsumer не реализует интерфейс Consumer, потому что accept может бросать InterruptedException.
 * <p>
 * AsyncConsumer.close() закрывает канал и ждет когда будут обработаны все накопившиеся объекты и завершится
 * тред-обработчик.
 * <p>
 * AsyncConsumer.cancel() закрывает канал, прерывает тред-обработчик и ждет когда он завершится.
 * Все необработанные элементы (включая тот, на котором обработка была прервана) попадают в failedItems.
 *
 * @param <T> - тип потребляемых объектов.
 */
public class AsyncConsumer<T> implements AutoCloseable {
    public static class AsyncConsumerException extends RuntimeException {
        public AsyncConsumerException(String message) {
            super(message);
        }

        public AsyncConsumerException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class FailedItems<T> {
        private final List<T> items;

        public FailedItems() {
            this.items = new ArrayList<>();
        }

        public void add(T item) {
            items.add(item);
        }

        private List<T> getItems() {
            return items;
        }
    }

    private final BlockingChannel<T> channel;
    private final Future<?> runner;
    private final FailedItems<T> failedItems;

    // Нужно отличать ситуации, когда канал закрылся из-за закрытия AsyncConsumer и когда
    // канал закрылся из-за неожиданного завершения consumer-треда.
    private volatile boolean closeInvoked;

    public AsyncConsumer(Interrupts.InterruptibleConsumer<T> consumer, int capacity) {
        this(consumer, capacity, null);
    }

    public AsyncConsumer(Interrupts.InterruptibleConsumer<T> consumer, int capacity, String threadNamePrefix) {
        this(makeChannelConsumer(consumer), capacity, threadNamePrefix);
    }

    public AsyncConsumer(
            Interrupts.InterruptibleBiConsumer<BlockingChannelReader<T>, FailedItems<T>> channelConsumer,
            int capacity,
            String threadNamePrefix
    ) {
        this(channelConsumer, capacity, threadNamePrefix, null);
    }

    public AsyncConsumer(
            Interrupts.InterruptibleBiConsumer<BlockingChannelReader<T>, FailedItems<T>> channelConsumer,
            int capacity,
            String threadNamePrefix,
            Queue<T> channelUnderlyingQueue
    ) {
        try (Transient<BlockingChannel<T>> holder = new Transient<>()) {
            closeInvoked = false;
            failedItems = new FailedItems<>();
            channel = holder.item = channelUnderlyingQueue == null
                    ? new BlockingChannel<>(capacity)
                    : new BlockingChannel<>(capacity, channelUnderlyingQueue);

            ExecutorService executor = Executors.newSingleThreadExecutor(new NamedThreadFactory(threadNamePrefix));
            try {
                runner = executor.submit((Callable<Void>) (() -> {
                    run(channel, failedItems, channelConsumer);
                    return null;
                }));
            } finally {
                executor.shutdown();
            }

            holder.success();
        }
    }

    public void accept(T item) throws InterruptedException {
        try {
            channel.put(item);
        } catch (BlockingChannel.ClosedChannelException exc) {
            handleClosedChannel();
        }
    }

    /**
     * Неблокирующая версия accept. Если в канале кончилось место, возвращает исключение.
     */
    public void acceptNonBlocking(T item) {
        try {
            channel.add(item);
        } catch (BlockingChannel.ClosedChannelException exc) {
            handleClosedChannel();
        }
    }

    @Override
    public void close() {
        closeInvoked = true;
        channel.close();
        waitRunner();
    }

    /**
     * @return true - если consumer-тред завершился.
     */
    public boolean isStopped() {
        return runner.isDone();
    }

    public void cancel() {
        // Нужно, чтобы InterruptedException прилетел в runner раньше чем закрытие очереди,
        // чтобы у треда была возможность отличить штатное завершеение и прерывание работы.
        runner.cancel(true);
        channel.close();
    }

    /**
     * Ждет закрытия консьюмера и завершения consumer-треда.
     *
     * @throws InterruptedException
     */
    @SuppressWarnings("squid:S1166") // осознано "проглатываем" исключения
    public void awaitStop(Duration timeout) throws InterruptedException {
        if (!isStopped()) {
            try {
                runner.get(timeout.toMillis(), TimeUnit.MILLISECONDS);
            } catch (ExecutionException | CancellationException | TimeoutException exc) {
                // Любое из перечисленных исключений означает завершение ожидания,
                // перевыбрасывать или логгировать их не нужно.
            }
        }
    }

    /**
     * @return список элементов, обработка которых не удалась из-за ошибки.
     */
    public synchronized List<T> getFailedItems() {
        if (!isStopped()) {
            throw new IllegalStateException("AsyncConsumer must be stopped before getting failed items");
        }
        return failedItems.getItems();
    }

    private void handleClosedChannel() {
        if (!channel.isClosed()) {
            throw new IllegalStateException("Internal error:" +
                    " channel must be closed before call to handleClosedChannel");
        }
        // Канал может быть закрыт по одной из следующих причин:
        if (closeInvoked) {
            // 1. Был вызван метод AsyncConsumer.close()
            //    Это значит, кто-то пытался воспользоваться консьюмером после его закрытия.
            //    Похоже, на ошибку в программе.
            throw new AsyncConsumerException("Consumer is closed");

        } else if (runner.isCancelled()) {
            // 2. Был вызван метод AsyncConsumer.cancel()
            //    Это значит, кто-то пытался воспользоваться консьюмером после его отмены.
            //    Похоже, на ошибку в программе.
            throw new AsyncConsumerException("Consumer is cancelled");

        } else {
            // 3. Сам consumer неожиданно завершился с ошибкой.
            //    В таком случае, runner thread уже завершился или будет вот-вот завершен
            //    (закрытие канала происходит непосредственно перед завершением треда,
            //    когда код channelConsumer-а уже завершен).
            //    Поэтому ждем завершения и репортим ошибку с cause = оригинальному исключению
            //    из треда.
            waitRunner();

            // 4. Сам consumer может неожиданно завершиться, но без ошибки
            //    (в этом случае waitRunner не бросит исключение)
            throw new AsyncConsumerException("Consumer thread is unexpectedly stopped without an error");

            // Есть еще один экзотический способ попасть в данную ветвь исполнения -
            // в результате гонки, если в процессе разматывания стека в consumer-треде кто-то
            // вызвал cancel(). В этом случае исходная ошибка будет безвозвратно потеряна.

            // Специально обрабаотывать данный случай нет особого смысла, потому что эта гонка
            // может случиться и после того как мы сообщили о неожиданной остановке треда.
        }
    }

    @SuppressWarnings("squid:S1166") // осознано "проглатываем" исключения
    private void waitRunner() {
        try {
            // Здесь мы потенциально можем зависнуть навечно. С одной стороны, это не страшно,
            // так как это может случиться только если runner завис навечно. А если runner завис навечно,
            // то, как ни крути, программа не завершится. С другой стороны, если бы был таймаут на runner.get,
            // то был бы шанс увидеть исключение в логе, даже при зависшей программе.
            //
            // Возможно, стоит поменять на criticalWaitTimeout.
            Interrupts.resurrectingRun(runner::get);
        } catch (CancellationException ignored) {
            // Считаем отмену нормальным завешрением
        } catch (ExecutionException exc) {
            throw new AsyncConsumerException("Consumer thread is failed", exc.getCause());
        }
    }

    private static <T> void run(
            BlockingChannel<T> channel,
            FailedItems<T> failedItems,
            Interrupts.InterruptibleBiConsumer<BlockingChannelReader<T>, FailedItems<T>> consumer
    )
            throws InterruptedException {
        try (BlockingChannel<T> ignored = channel) {
            consumer.accept(new BlockingChannelReader<>(channel), failedItems);
        } finally {
            Optional<T> next;
            while ((next = channel.next()).isPresent()) {
                failedItems.add(next.get());
            }
        }
    }

    private static <T> Interrupts.InterruptibleBiConsumer<BlockingChannelReader<T>, FailedItems<T>> makeChannelConsumer(
            Interrupts.InterruptibleConsumer<T> consumer
    ) {
        return (channel, failedItems) -> {
            Optional<T> optionalNext;
            while ((optionalNext = channel.next()).isPresent()) {
                T next = optionalNext.get();
                try {
                    consumer.accept(next);
                } catch (RuntimeException | Error exc) {
                    failedItems.add(next);
                    throw exc;
                }
            }
        };
    }
}
