package ru.yandex.direct.utils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Stream;

/**
 * Список AutoCloseable элементов. Сам список тоже AutoCloseable. При закрытии пытается закрыть все свои элементы.
 *
 * @param <T> тип элементов
 */
public class AutoCloseableList<T extends AutoCloseable> implements AutoCloseable, Iterable<T> {
    private final List<T> items;

    public AutoCloseableList() {
        this.items = new ArrayList<>();
    }

    public synchronized void add(Transient<T> item) {
        items.add(item.item);
        item.success();
    }

    public int size() {
        return items.size();
    }

    public Stream<T> stream() {
        return items.stream();
    }

    @Override
    public Iterator<T> iterator() {
        return items.iterator();
    }

    @Override
    public void close() {
        forcedForEach(items, AutoCloseable::close, new IllegalStateException("Some items have failed to close"));
    }

    public static <T, E extends Exception> void forcedForEach(
            Collection<T> items,
            Checked.CheckedConsumer<T, E> consumer,
            RuntimeException rootException
    ) {
        boolean thrown = false;

        for (T item : items) {
            try {
                consumer.accept(item);
            } catch (Exception exc) {
                thrown = true;
                rootException.addSuppressed(exc);
            }
        }

        if (thrown) {
            throw rootException;
        }
    }
}
