package ru.yandex.direct.utils;

import java.time.Duration;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Это как BlockingQueue, только с возможностью закрытия.
 * <p>
 * После закрытия канала в него больше нельзя писать. Попытка записи в закрытый канал бросает исключение.
 * <p>
 * Чтение из закрытого канала разрешено.
 * Если в закрытом канале нет элементов, take() бросает исключение, а next() возвращает Optional.empty().
 *
 * @param <T> тип элементов канала.
 */
public class BlockingChannel<T> implements AutoCloseable {
    public static class EndOfChannel extends RuntimeException {
        public EndOfChannel() {
            super("Channel is closed and empty");
        }
    }

    public static class ClosedChannelException extends RuntimeException {
        public ClosedChannelException() {
            super("Attempted to write to closed channel");
        }
    }

    private final Lock lock;
    private final Condition canRead;
    private final Condition canWrite;
    private final Condition closedCondition;
    private final Queue<T> queue;
    private final int capacity;
    private volatile boolean closed;

    public BlockingChannel(int capacity) {
        this(capacity, new ArrayDeque<>(capacity));
    }

    /**
     * @param capacity        вместимость канала, максимальное количество элементов, хранимое в очереди канала
     * @param underlyingQueue Очередь, которая будет использоваться для хранения элементов.
     *                        Очередь может быть потоко-небезопасной, всю синхронизацию берет на себя канал.
     */
    public BlockingChannel(int capacity, Queue<T> underlyingQueue) {
        if (capacity <= 0) {
            throw new IllegalArgumentException("Capacity must be greater than zero");
        }
        this.lock = new ReentrantLock();
        this.canRead = lock.newCondition();
        this.canWrite = lock.newCondition();
        this.closedCondition = lock.newCondition();
        this.queue = underlyingQueue;
        this.capacity = capacity;
        this.closed = false;
    }

    public BlockingChannel(Collection<T> items) {
        this(items.size());
        for (T item : items) {
            add(item);
        }
    }

    public void put(T item) throws InterruptedException {
        lock.lock();
        try {
            while (!closed && queue.size() >= capacity) {
                canWrite.await();
            }
            if (closed) {
                throw new ClosedChannelException();
            }
            queue.add(item);
            canRead.signal();
        } finally {
            lock.unlock();
        }
    }

    public void add(T item) {
        lock.lock();
        try {
            if (closed) {
                throw new ClosedChannelException();
            }
            if (queue.size() >= capacity) {
                throw new IllegalStateException("BlockingChannel is full");
            }
            queue.add(item);
            canRead.signal();
        } finally {
            lock.unlock();
        }
    }

    public T take() throws InterruptedException {
        return next().orElseGet(() -> {
            throw new EndOfChannel();
        });
    }

    /**
     * @return пусто, если очередь пуста.
     */
    public Optional<T> poll() {
        lock.lock();
        try {
            if (queue.isEmpty()) {
                return Optional.empty();
            } else {
                T item = queue.remove();
                canWrite.signal();
                return Optional.of(item);
            }
        } finally {
            lock.unlock();
        }
    }

    /**
     * @return пусто, если канал закрыт и пуст. Блокитует исполнение, если канал пуст но не закрыт.
     */
    public Optional<T> next() throws InterruptedException {
        lock.lock();
        try {
            while (!closed && queue.isEmpty()) {
                canRead.await();
            }
            // Позволяем дочитать накопившиеся данные, даже если канал закрыли
            return poll();
        } finally {
            lock.unlock();
        }
    }

    /**
     * @return пусто, если канал закрыт и пуст. Блокитует исполнение, если канал пуст но не закрыт.
     */
    public Optional<T> next(Duration timeout) throws InterruptedException {
        long nanos = timeout.toNanos();
        lock.lock();
        try {
            while (!closed && queue.isEmpty() && nanos > 0) {
                nanos = canRead.awaitNanos(nanos);
            }
            return poll();
        } finally {
            lock.unlock();
        }
    }

    /**
     * @return все накопившиеся в канале объекты, если нет ни одного - ждет.
     * Если канал закрыт и пуст - возвращает пусто.
     * Пустой список может вернутся тогда и только тогда, когда канал закрыт и пуст.
     * @throws InterruptedException
     */
    public List<T> takeAll() throws InterruptedException {
        return takeAllInto(new ArrayList<>());
    }

    /**
     * @param collection куда писать элементы
     * @param <C>        тип коллекции
     * @return collection, возможно, дополненная новыми элементами.
     * <p>
     * Если collection не пуста, а в канале на момент вызова метода нет элементов,
     * то вызов блокироваться не будет, а collection будет возвращена без изменений.
     * <p>
     * Коллекция будет пуста только если канал закрыт и пуст и в коллекции изначально не было элементов.
     * <p>
     * Будьте бдительны. Если канал закрыт и пуст, а переданная в параметрах коллекция не пуста, то
     * она будет возвращена без изменений. Так можно по ошибке прозевать момент закрытия канала и уйти в
     * вечный цикл.
     * @throws InterruptedException
     */
    public <C extends Collection<T>> C takeAllInto(C collection) throws InterruptedException {
        lock.lock();
        try {
            while (!closed && queue.isEmpty() && collection.isEmpty()) {
                canRead.await();
            }
            while (!queue.isEmpty()) {
                collection.add(queue.remove());
            }
            return collection;
        } finally {
            lock.unlock();
        }
    }

    /**
     * @param size возвращать не больше чем size элементов.
     * @return накопившиеся в канале объекты, но не больше чем size. Если нет ни одного - ждет.
     * Если канал закрыт и пуст - возвращает пусто.
     * Пустой список может вернутся тогда и только тогда, когда канал закрыт и пуст.
     * @throws InterruptedException
     */
    public List<T> takeBatch(int size) throws InterruptedException {
        return takeBatchInto(new ArrayList<>(), size);
    }

    /**
     * @param collection куда писать элементы
     * @param size       писать элементы в collection, пока ее размер не превышает size
     * @param <C>        тип коллекции
     * @return collection, возможно, дополненная новыми элементами.
     * <p>
     * Если collection не пуста, а в канале на момент вызова метода нет элементов,
     * то вызов блокироваться не будет, а collection будет возвращена без изменений.
     * <p>
     * Коллекция будет пуста только если канал закрыт и пуст и в коллекции изначально не было элементов.
     * <p>
     * Будьте бдительны. Если канал закрыт и пуст, а переданная в параметрах коллекция не пуста, то
     * она будет возвращена без изменений. Так можно по ошибке прозевать момент закрытия канала и уйти в
     * вечный цикл.
     * @throws InterruptedException
     */
    public <C extends Collection<T>> C takeBatchInto(C collection, int size) throws InterruptedException {
        if (size <= 0) {
            throw new IllegalArgumentException("size must be positive, got " + size);
        }
        lock.lock();
        try {
            while (!closed && queue.isEmpty() && collection.isEmpty()) {
                canRead.await();
            }
            while (!queue.isEmpty() && collection.size() < size) {
                collection.add(queue.remove());
            }
            return collection;
        } finally {
            lock.unlock();
        }
    }

    public boolean isEmpty() {
        lock.lock();
        try {
            return queue.isEmpty();
        } finally {
            lock.unlock();
        }
    }

    public boolean isClosed() {
        lock.lock();
        try {
            return closed;
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void close() {
        lock.lock();
        try {
            if (!closed) {
                closed = true;
                canRead.signalAll();
                canWrite.signalAll();
                closedCondition.signalAll();
            }
        } finally {
            lock.unlock();
        }
    }

    public void awaitClose(Duration timeout) throws InterruptedException {
        long nanos = timeout.toNanos();
        lock.lock();
        try {
            while (!closed && nanos > 0) {
                nanos = closedCondition.awaitNanos(nanos);
            }
        } finally {
            lock.unlock();
        }
    }
}
