package ru.yandex.direct.utils;

import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Обертки для стандартных функциональных интерфейсов, поддерживающие checked-исключения.
 */
public interface Checked {
    @FunctionalInterface
    interface CheckedConsumer<T, E extends Exception> {
        void accept(T a) throws E;
    }

    @FunctionalInterface
    interface CheckedFunction<T, R, E extends Exception> {
        R apply(T t) throws E;
    }

    @FunctionalInterface
    interface CheckedRunnable<E extends Exception> {
        void run() throws E;
    }

    @FunctionalInterface
    interface CheckedSupplier<T, E extends Exception> {
        T get() throws E;
    }

    class CheckedException extends RuntimeException {
        public CheckedException(Throwable cause) {
            super(cause);
        }

        public CheckedException(String message, Throwable cause) {
            super(message, cause);
        }

        public Throwable unwrapped() {
            Throwable unwrapped = getCause();
            for (Throwable suppressed : getSuppressed()) {
                unwrapped.addSuppressed(suppressed);
            }
            return unwrapped;
        }
    }

    /**
     * @param function - функция, бросающая checked-исключения.
     * @return функция, оборачивающая checked-исключение вылетевшее из function в unchecked-исключение.
     * <p>
     * Пример использования:
     * <p>
     * List\<Path\> symlinkTargets = symlinks.stream()
     * .map(Checked.function(Files::readSymbolicLink))
     * .collect(Collectors.toList());
     */
    static <T, R, E extends Exception> Function<T, R> function(CheckedFunction<T, R, E> function) {
        return a -> {
            try {
                return function.apply(a);
            } catch (Exception exc) {
                throwWrapped(exc);
                // Следующий throw нужен, чтобы идея не ругалась на Missing return statement.
                throw new IllegalStateException("Unreachable code");
            }
        };
    }

    /**
     * Аналогично function, только для Supplier\<T\>
     */
    static <T, E extends Exception> Supplier<T> supplier(CheckedSupplier<T, E> supplier) {
        return () -> get(supplier);
    }

    /**
     * Аналогично function, только для Consumer\<T\>
     */
    static <T, E extends Exception> Consumer<T> consumer(CheckedConsumer<T, E> consumer) {
        return a -> {
            try {
                consumer.accept(a);
            } catch (Exception exc) {
                throwWrapped(exc);
            }
        };
    }

    /**
     * Аналогично function, только для Runnable\<T\>
     */
    static <E extends Exception> Runnable runnable(CheckedRunnable<E> runnable) {
        return () -> run(runnable);
    }

    /**
     * Выполняет runnable, оборачивая checked-исключения в unchecked
     */
    static <E extends Exception> void run(CheckedRunnable<E> runnable) {
        try {
            runnable.run();
        } catch (Exception exc) {
            throwWrapped(exc);
        }
    }

    /**
     * Выполняет supplier.get(), оборачивая checked-исключения в unchecked
     */
    static <T, E extends Exception> T get(CheckedSupplier<T, E> supplier) {
        try {
            return supplier.get();
        } catch (Exception exc) {
            throwWrapped(exc);
            // Следующий throw нужен, чтобы идея не ругалась на Missing return statement.
            throw new IllegalStateException("Unreachable code");
        }
    }

    static void throwWrapped(Throwable exc) {
        if (exc instanceof Error) {
            throw (Error) exc;
        } else if (exc instanceof RuntimeException) {
            throw (RuntimeException) exc;
        } else if (exc instanceof InterruptedException) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(exc);
        } else {
            throw new CheckedException(exc);
        }
    }
}
