package ru.yandex.direct.utils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import static java.util.stream.Collectors.toSet;

@ParametersAreNonnullByDefault
public class CollectionUtils {
    private CollectionUtils() {
    }

    /**
     * Проверяет, что все переданные коллекции пустые или null
     */
    public static boolean isAllEmpty(Collection... collections) {
        return Stream.of(collections)
                .allMatch(collection -> collection == null || collection.isEmpty());
    }

    /**
     * Проверяет, что коллекция пустая или null
     */
    public static boolean isEmpty(@Nullable Collection collection) {
        return collection == null || collection.isEmpty();
    }

    /**
     * Проверяет, что коллекция не пустая и не null
     */
    public static boolean isNotEmpty(@Nullable Collection collection) {
        return !isEmpty(collection);
    }

    /**
     * @return первый аргумент, если он не null и не isEmpty, иначе второй
     */
    @Nullable
    public static <T extends Collection> T defaultIfEmpty(@Nullable T coll, @Nullable T defaultColl) {
        return isEmpty(coll) ? defaultColl : coll;
    }

    /**
     * @return первый аргумент, если он не null и не isEmpty, иначе null
     */
    @Nullable
    public static <T extends Collection> T nullIfEmpty(@Nullable T coll) {
        return isEmpty(coll) ? null : coll;
    }

    public static <T> List<T> flatToList(Collection<? extends Collection<T>> collections) {
        List<T> result = new ArrayList<>();
        collections.forEach(result::addAll);
        return result;
    }

    public static <T> Set<T> flatToSet(Collection<? extends Collection<T>> collections) {
        Set<T> result = new HashSet<>();
        collections.forEach(result::addAll);
        return result;
    }

    @SafeVarargs
    public static <T> Set<T> orderedSet(T... elements) {
        return new LinkedHashSet<>(Arrays.asList(elements));
    }

    @SafeVarargs
    public static <T> Set<T> unionToSet(Collection<T>... elements) {
        return Stream.of(elements)
                .flatMap(Collection::stream)
                .collect(toSet());
    }

}
