package ru.yandex.direct.utils;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.BitSet;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.commons.lang3.mutable.MutableObject;

public class CommonUtils {
    private CommonUtils() {
    }

    /**
     * Упрощённая запись Optional.ofNullable(first).orElse(second);
     *
     * @return первый аргумент, если он не null, иначе второй
     */
    @Nonnull
    public static <T> T nvl(@Nullable T first, @Nonnull T second) {
        return first != null ? first : Objects.requireNonNull(second);
    }

    /**
     * Упрощённая запись Optional.ofNullable(first).orElseGet(secondSupplier);
     */
    @Nonnull
    public static <T> T nvl(@Nullable T first, @Nonnull Supplier<T> secondSupplier) {
        return first != null ? first : Objects.requireNonNull(secondSupplier.get());
    }

    /**
     * Упрощённая запись Optional.ofNullable(first).orElse(second);
     * Возможна передача null вторым параметром
     *
     * @return первый аргумент, если он не null, иначе второй
     */
    @Nullable
    public static <T> T nullableNvl(@Nullable T first, @Nullable T second) {
        return first != null ? first : second;
    }

    /**
     * Вычислить значение из аргумента, если он не равен {@code null}.
     */
    @Nullable
    public static <T, X> X ifNotNull(@Nullable T value, @Nonnull Function<T, X> converter) {
        return value != null ? converter.apply(value) : null;
    }

    /**
     * Вычислить значение из аргумента, если он не равен {@code null}, иначе вернуть третий аргумент.
     */
    @Nonnull
    public static <T, X> X ifNotNullOrDefault(@Nullable T value, @Nonnull Function<T, X> converter,
                                              @Nonnull X byDefault) {
        return value != null ? converter.apply(value) : byDefault;
    }

    /**
     * Безопасное приведение типов, возвращающее null в случае неудачи.
     */
    @Nullable
    public static <T> T safeCast(@Nullable Object value, Class<T> clazz) {
        return clazz.isInstance(value) ? clazz.cast(value) : null;
    }

    /**
     * Проверить, что переданное значение может быть числовым идентификатором, т.е. задано и положительно
     */
    public static boolean isValidId(@Nullable Long value) {
        return value != null && value > 0L;
    }

    /**
     * Собрать стрим в мапу: stream().collect(mapByKey(t->t.getKey()));
     *
     * @param keyField функция получения ключа для мапы
     * @param <T>      Тип 1го элемента стрима
     * @return collector
     */
    @Nonnull
    public static <T> Collector<T, ?, Map<Long, T>> mapByKey(Function<T, Long> keyField) {
        return Collectors.toMap(keyField, Function.identity());
    }

    /**
     * Простая мемоизация, не thread-safe
     */
    public static <T> Supplier<T> memoize(Supplier<T> delegate) {
        MutableObject<T> cache = new MutableObject<>();
        return () -> {
            if (cache.getValue() == null) {
                cache.setValue(Objects.requireNonNull(delegate.get()));
            }
            return cache.getValue();
        };
    }

    /**
     * Простая мемоизация для функций, не thread-safe, запоминает на всё время жизни возвращаемого объекта.
     */
    public static <T, R> Function<T, R> memoize(Function<T, R> delegate) {
        return memoize(delegate, new HashMap<>());
    }

    /**
     * Простая мемоизация для функций. Будет thread-safe если оба аргумента thread-safe.
     *
     * @param memory Объект, в котором будут храниться результаты.
     */
    private static <T, R> Function<T, R> memoize(Function<T, R> delegate, Map<T, R> memory) {
        return arg -> memory.computeIfAbsent(arg, delegate);
    }

    /**
     * Thread-safe мемоизация
     */
    public static <T> Supplier<T> memoizeLock(Supplier<T> delegate) {
        AtomicReference<T> value = new AtomicReference<>();
        return () -> {
            T val = value.get();
            if (val == null) {
                synchronized (value) {
                    val = value.get();
                    if (val == null) {
                        val = Objects.requireNonNull(delegate.get());
                        value.set(val);
                    }
                }
            }
            return val;
        };
    }

    /**
     * Returns flags of duplicated elements (true - element is duplicated).
     * If {@code comparator} is {@code null} then {@link Object#equals(Object)} is used to detect duplicates.
     * In this case each list item must contain valid methods {@code equals()} and {@code hashCode()}.
     * If {@code comparator} is not {@code null} it will be used to compare objects.
     *
     * @param list       list of objects to be checked for containing duplicates
     * @param comparator if not {@code null} it will be used for duplicate searching. If {@code null} objects
     *                   are compared by {@link Object#equals(Object)}
     * @return {@link BitSet} where set bits are placed in positions of items with duplicates
     * @throws NullPointerException if {@code comparator} is set and it doesn't accept {@code null}-values
     * @see #getDuplicatedItemsFlags(List)
     */
    public static <T> BitSet getDuplicatedItemsFlags(List<T> list, @Nullable Comparator<T> comparator)
            throws NullPointerException {
        BitSet duplicatedItemsFlags = new BitSet(list.size());
        Map<T, Integer> itemsIndexesMap;
        if (comparator != null) {
            itemsIndexesMap = new TreeMap<>(comparator);
        } else {
            itemsIndexesMap = new HashMap<>();
        }

        for (int i = 0; i < list.size(); i++) {
            T curItem = list.get(i);
            if (curItem == null) {
                continue;
            }
            Integer prevIndex;
            if ((prevIndex = itemsIndexesMap.put(curItem, i)) != null) {
                duplicatedItemsFlags.set(prevIndex);
                duplicatedItemsFlags.set(i);
            }
        }

        return duplicatedItemsFlags;
    }

    /**
     * Returns flags of duplicated elements (true - element is duplicated).
     * Each list item must contain valid methods {@code equals()} and {@code hashCode()}.
     *
     * @param list list of objects of any type but containing valid methods
     *             {@code equals()} and {@code hashCode()}. Can contain {@code null}-values.
     * @see #getDuplicatedItemsFlags(List, Comparator)
     */
    public static BitSet getDuplicatedItemsFlags(List<?> list) {
        return getDuplicatedItemsFlags(list, null);
    }

    public static <T> Stream<T> stream(Iterator<T> iterator) {
        return StreamSupport.stream(
                Spliterators.spliteratorUnknownSize(iterator, Spliterator.ORDERED),
                false
        );
    }

    @SafeVarargs
    public static <T extends Comparable<? super T>> T min(T first, T... items) {
        T min = first;
        for (T item : items) {
            if (min.compareTo(item) > 0) {
                min = item;
            }
        }
        return min;
    }

    @SafeVarargs
    public static <T extends Comparable<? super T>> T max(T first, T... items) {
        T max = first;
        for (T item : items) {
            if (max.compareTo(item) < 0) {
                max = item;
            }
        }
        return max;
    }

    public static ChronoUnit chronoUnit(TimeUnit unit) {
        switch (unit) {
            case NANOSECONDS:
                return ChronoUnit.NANOS;
            case MICROSECONDS:
                return ChronoUnit.MICROS;
            case MILLISECONDS:
                return ChronoUnit.MILLIS;
            case SECONDS:
                return ChronoUnit.SECONDS;
            case MINUTES:
                return ChronoUnit.MINUTES;
            case HOURS:
                return ChronoUnit.HOURS;
            case DAYS:
                return ChronoUnit.DAYS;
            default:
                throw new IllegalArgumentException("Unknown TimeUnit constant: " + unit);
        }
    }

    public static String formatApproxDuration(Duration duration) {
        if (duration.toHours() > 0) {
            return String.format(
                    "%d:%02d:%02ds",
                    duration.toHours(),
                    duration.toMinutes() % 60,
                    duration.getSeconds() % 60
            );
        } else if (duration.toMinutes() > 0) {
            return String.format(
                    "%d:%02ds",
                    duration.toMinutes() % 60,
                    duration.getSeconds() % 60
            );
        } else {
            return String.format("%ds", duration.getSeconds() % 60);
        }
    }

    /**
     * Returns empty list if argument is null, otherwise returns list with value.
     *
     * @param value Content of list, if not null
     * @param <V>   any type
     * @return List with zero or one element
     */
    public static <V> List<V> listOfNullable(@Nullable V value) {
        return value == null ? Collections.emptyList() : Collections.singletonList(value);
    }

    /**
     * Сжимает значения в мапе. Числа по одинаковым индексам останутся одинаковыми, но их значения будут
     * от 1 до количества различных чисел среди значений входящей мапы.
     * Например: (4->5; 10->37; 11->5; 14->2) => (4->1; 10->2; 11->1; 14->3)
     */
    public static Map<Integer, Integer> compressNumbers(Map<Integer, Long> numbers) {
        Map<Long, Integer> numberToCompressedNumber = new HashMap<>();
        Map<Integer, Integer> compressedMap = new HashMap<>();
        numbers.forEach((index, number) -> {
            numberToCompressedNumber.putIfAbsent(number, numberToCompressedNumber.size() + 1);
            compressedMap.put(index, numberToCompressedNumber.get(number));
        });
        return compressedMap;
    }

    /**
     * Синтаксический сахар, возвращает результат противоположенный методу {@link
     * java.util.Objects#equals}.
     */
    public static boolean notEquals(Object a, Object b) {
        return !Objects.equals(a, b);
    }

    @Nullable
    public static <T> T coalesce(@Nullable T a, @Nullable T b, @Nullable T c) {
        return a != null ? a : (b != null ? b : c);
    }

}
