package ru.yandex.direct.utils;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class Completer implements AutoCloseable {
    public static class Builder {
        private final Duration closeTimeout;
        private final List<CompleterFuture<?>> tasks;
        private final BookKeeper bookKeeper;

        public Builder(Duration closeTimeout) {
            this.closeTimeout = closeTimeout;
            this.tasks = new ArrayList<>();
            this.bookKeeper = new BookKeeper();
        }

        public <E extends Exception> Future<Void> submitVoid(String taskName, Checked.CheckedRunnable<E> runnable) {
            return submit(taskName, () -> {
                runnable.run();
                return null;
            });
        }

        public <T> Future<T> submit(String taskName, Callable<T> callable) {
            CompleterFuture<T> future = new CompleterFuture<>(taskName, callable, bookKeeper);
            tasks.add(future);
            return future;
        }

        public Completer build() {
            return new Completer(closeTimeout, tasks, bookKeeper);
        }
    }

    private final ExecutorService executor;
    private final Duration closeTimeout;
    private final List<CompleterFuture<?>> tasks;
    private final BookKeeper bookKeeper;

    private Completer(Duration closeTimeout, List<CompleterFuture<?>> tasks, BookKeeper bookKeeper) {
        this.closeTimeout = closeTimeout;
        this.bookKeeper = bookKeeper;
        this.tasks = new ArrayList<>();
        this.executor = new Executor(bookKeeper);
        try (Transient<Completer> holder = new Transient<>(this)) {
            for (CompleterFuture<?> task : tasks) {
                this.executor.execute(task);
                this.tasks.add(task);
            }
            holder.success();
        }
    }

    public void waitAll() throws InterruptedException {
        bookKeeper.waitCompletionOrFailure(tasks.size());
    }

    public boolean waitAll(Duration timeout) throws InterruptedException {
        return bookKeeper.waitCompletionOrFailure(tasks.size(), timeout);
    }

    @Override
    public void close() {
        executor.shutdownNow();
        Interrupts.waitUninterruptibly(closeTimeout, t -> this.bookKeeper.waitAllFinished(tasks.size(), t));
        AutoCloseableList.forcedForEach(
                tasks,
                this::reportTaskError,
                new CompleterException("Some tasks have failed")
        );
    }

    @SuppressWarnings("squid:S1166") // Игнорируем CancellationException, а у ExecutionException берем только cause
    private <T> void reportTaskError(CompleterFuture<T> task) {
        if (!task.isDone()) {
            throw new CompleterException(String.format(
                    "Task '%s' is still running, interruption is timed out (timeout=%s)",
                    task.getName(),
                    CommonUtils.formatApproxDuration(closeTimeout)
            ));
        }

        // Если пользователь отменил задачу по cancel(), то формально она done,
        // но не факт, что выполнение задачи завершилось.
        if (task.isRunning()) {
            throw new CompleterException(String.format(
                    "Cancelled task '%s' is still running, interruption is timed out (timeout=%s)",
                    task.getName(),
                    CommonUtils.formatApproxDuration(closeTimeout)
            ));
        }

        try {
            Interrupts.resurrectingRun(task::get);
        } catch (CancellationException exc) {
            // не репортим отмену
        } catch (ExecutionException execExc) {
            Throwable exc = execExc.getCause();
            if (exc instanceof InterruptedException) {
                // Результат вызова executor.shutdownNow(). Не репортим, как и в случае cancel(),
                // кроме случаев, когда прерывание повлекло за собой другие исключения.
                if (exc.getCause() != null || exc.getSuppressed().length != 0) {
                    throw new CompleterException(
                            String.format("Task '%s' is failed when interrupted", task.getName()),
                            exc
                    );
                }
            } else {
                throw new CompleterException(
                        String.format("Task '%s' is failed with exception", task.getName()),
                        exc
                );
            }
        }
    }

    public static class CompleterException extends RuntimeException {
        CompleterException(String message) {
            super(message);
        }

        CompleterException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    private static class CompleterFuture<T> extends FutureTask<T> {
        private final String name;
        private final BookKeeper bookKeeper;
        private volatile boolean running;

        private CompleterFuture(String name, Callable<T> callable, BookKeeper bookKeeper) {
            super(callable);
            this.name = name;
            this.bookKeeper = bookKeeper;
            this.running = false;
        }

        String getName() {
            return name;
        }

        boolean isRunning() {
            return running;
        }

        private void setRunning(boolean running) {
            this.running = running;
        }

        @Override
        protected void done() {
            super.done();
            if (isCancelled()) {
                bookKeeper.taskDone();
            } else {
                setRunning(false);
                try {
                    Interrupts.resurrectingRun(this::get);
                    bookKeeper.taskDone();
                } catch (ExecutionException exc) {
                    bookKeeper.taskFailed(getName(), exc);
                }
            }
        }
    }

    private static class Executor extends ThreadPoolExecutor {
        private final BookKeeper bookKeeper;

        Executor(BookKeeper bookKeeper) {
            super(0, Integer.MAX_VALUE, 0, TimeUnit.SECONDS, new SynchronousQueue<>());
            this.bookKeeper = bookKeeper;
        }

        @Override
        protected void beforeExecute(Thread t, Runnable r) {
            super.beforeExecute(t, r);
            if (r instanceof CompleterFuture) {
                CompleterFuture future = (CompleterFuture<?>) r;
                t.setName(future.getName());
                future.setRunning(true);
            } else {
                throw new IllegalArgumentException("Expected CompleterFuture, got: " + r);
            }
        }

        @Override
        protected void afterExecute(Runnable r, Throwable t) {
            super.afterExecute(r, t);
            if (r instanceof CompleterFuture) {
                CompleterFuture future = (CompleterFuture<?>) r;
                future.setRunning(false);
                bookKeeper.taskFinished();
            } else {
                throw new IllegalArgumentException("Expected CompleterFuture, got: " + r);
            }
        }
    }

    private static class BookKeeper {
        // количество задач, которые либо завершились либо были отменены
        private int done;
        // количество задач, которые завершились, т.е. соответствующий callable закончил исполнение
        private int finished;
        private String firstFailedTaskName;
        private Throwable firstFailedTaskException;

        BookKeeper() {
            done = 0;
            finished = 0;
            firstFailedTaskName = null;
            firstFailedTaskException = null;
        }

        synchronized void taskFinished() {
            finished += 1;
            notifyAll();
        }

        synchronized void taskDone() {
            done += 1;
            notifyAll();
        }

        synchronized void taskFailed(String failedTaskName, ExecutionException failedTaskException) {
            done += 1;
            if (firstFailedTaskName == null) {
                firstFailedTaskName = failedTaskName;
                firstFailedTaskException = failedTaskException.getCause();
            }
            notifyAll();
        }

        synchronized void waitCompletionOrFailure(int minTasksCount) throws InterruptedException {
            while (done < minTasksCount && firstFailedTaskException == null) {
                wait();
            }
            if (firstFailedTaskException != null) {
                throw new CompleterException("Failed task '" + firstFailedTaskName + "'", firstFailedTaskException);
            }
        }

        synchronized boolean waitCompletionOrFailure(int minTasksCount, Duration timeout) throws InterruptedException {
            MonotonicTime now = NanoTimeClock.now();
            MonotonicTime deadline = now.plus(timeout);
            while (done < minTasksCount && firstFailedTaskException == null) {
                if (now.isAtOrAfter(deadline)) {
                    return false;
                }
                Interrupts.waitMillisNanos(this::wait).await(deadline.minus(now));
                now = NanoTimeClock.now();
            }
            if (firstFailedTaskException != null) {
                throw new CompleterException("Failed task '" + firstFailedTaskName + "'", firstFailedTaskException);
            }
            return true;
        }

        synchronized boolean waitAllFinished(int minTasksCount, Duration timeout) throws InterruptedException {
            MonotonicTime now = NanoTimeClock.now();
            MonotonicTime deadline = now.plus(timeout);
            while (finished < minTasksCount) {
                if (now.isAtOrAfter(deadline)) {
                    return false;
                }
                Interrupts.waitMillisNanos(this::wait).await(deadline.minus(now));
                now = NanoTimeClock.now();
            }
            return true;
        }
    }
}
