package ru.yandex.direct.utils;


import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;

import javax.annotation.Nullable;

public class DateTimeUtils {

    public static final String MOSCOW_TIMEZONE = "Europe/Moscow";

    /**
     * Московская таймзона, в которых работает Директ и БК
     */
    public static final ZoneId MSK = ZoneId.of(MOSCOW_TIMEZONE);

    private DateTimeUtils() {
    }

    /**
     * @param timestamp время в секундах от начала эпохи
     * @return Локальное время {@link LocalDateTime}.
     */
    public static LocalDateTime fromEpochSeconds(Long timestamp) {
        return Instant.ofEpochSecond(timestamp)
                .atZone(ZoneId.systemDefault())
                .toLocalDateTime();
    }

    /**
     * @param timestamp время в миллисекундах от начала эпохи
     * @return Локальное время {@link LocalDateTime}.
     */
    public static LocalDateTime fromEpochMillis(Long timestamp) {
        return Instant.ofEpochMilli(timestamp)
                .atZone(ZoneId.systemDefault())
                .toLocalDateTime();
    }

    /**
     * @return Текущее время в секундах от начала эпохи.
     */
    public static long getNowEpochSeconds() {
        return Instant.now()
                .atZone(ZoneId.systemDefault())
                .toEpochSecond();
    }

    /**
     * @return время в секундах от начала эпохи
     */
    public static long moscowDateTimeToEpochSecond(LocalDateTime localDateTime) {
        return localDateTime.atZone(MSK).toEpochSecond();
    }

    /**
     * Перевести LocalDateTime по Московскому часовому поясу в UnixTime
     */
    public static Instant moscowDateTimeToInstant(LocalDateTime dateTime) {
        return dateTime.atZone(MSK).toInstant();
    }

    /**
     * Перевести UnixTime в LocalDateTime по Московской временной зоне
     */
    public static LocalDateTime instantToMoscowDateTime(Instant instant) {
        return instant.atZone(MSK).toLocalDateTime();
    }

    /**
     * Перевести UnixTime в LocalDate по Московской временной зоне
     */
    public static LocalDate instantToMoscowDate(Instant instant) {
        return instant.atZone(MSK).toLocalDate();
    }

    public static Long startOfDayInMoscowTime(LocalDate date) {
        return moscowDateTimeToInstant(date.atStartOfDay()).getEpochSecond();
    }

    /**
     * Аналог перлового метода TimeCommon::today()
     *
     * @return возвращает строку с текущей датой в формате YYYYMMDD
     */
    public static String today() {
        return LocalDateTime.now().format(DateTimeFormatter.BASIC_ISO_DATE);
    }

    public static boolean inFuture(@Nullable LocalDate date) {
        return date != null
                && LocalDate.now().isBefore(date);
    }

    public static boolean inFutureOrToday(@Nullable LocalDate date) {
        return date != null
                && !LocalDate.now().isAfter(date);
    }

    public static boolean inPast(@Nullable LocalDate date) {
        return date != null
                && date.isBefore(LocalDate.now());
    }

    public static boolean inPastOrToday(@Nullable LocalDate date) {
        return date != null
                && !date.isAfter(LocalDate.now());
    }

    public static LocalDateTime atTheBeginningOfMonth(LocalDate date) {
        return date.withDayOfMonth(1).atStartOfDay();
    }

}
