package ru.yandex.direct.utils;

import java.time.Duration;

/**
 * Заглушка-замена для NanoTimeClock в тестах.
 * С ее помощью можно тестировать правильность таймаутов.
 * <p>
 * Пример:
 * <p>
 * Есть метод, который несколько раз дергает некоторую http-ручку. Чтобы не нагружать сервер, требуется
 * дергать ручку не чаще чем раз в T миллисекунд. В тестах хочется проверить, что правильно выдерживается
 * пауза между последовательными http-запросами.
 * <p>
 * Какие есть варианты без FakeMonotonicClock:
 * <p>
 * - выставить в тестах маленькое значение T и замерить время за которое выполнился метод. Зная количество
 * выполненных http-зарпосов, можно прикинуть время за которое они должны были выполниться с учетом пауз.
 * Здесь проблема в том, что если код метода или http-ручка тормозят, это нельзя отличить от правильно
 * выдержанной паузы.
 * <p>
 * - выставить в тестах большое значение T, но тогда тесты будут выполняться неоправданно долго.
 * <p>
 * Этот класс позволяет притвориться, что на выполнение любого кода между последовательными вызовами getTime()
 * (не считая вызова FakeMonotonicClock.sleep()) тратится ноль времени. А вызов FakeMonotonicClock.sleep()
 * просто переводит часы на указанное время вперед.
 * <p>
 * Кроме этого, есть возможность синхронизировать течение времени между тредами,
 * для этого нужно выбрать тред, время в коротом будет считаться эталонным (см. referenceTimeline()).
 * Во всех остальных тредах FakeMonotonicClock.sleep() будет блокироваться до тех пор, пока указанное
 * количество фейкового времени не пройдет в эталонном треде.
 * <p>
 * Т.е. если тред A считается эталонным, то в треде B вызов FakeMonotonicClock.sleep(Duration.ofSeconds(5))
 * гарантировано завершиться не раньше чем тред A вызовет sleep суммарно на время не менее 5 секунд,
 * типа:
 * fakeClock.sleep(Duration.ofSeconds(3));
 * doSomething();
 * fakeClock.sleep(Duration.ofSeconds(2));
 */
public class FakeMonotonicClock implements MonotonicClock {
    public class ReferenceTimeline implements AutoCloseable {
        private boolean closed = false;

        @Override
        public void close() {
            if (!closed) {
                synchronized (FakeMonotonicClock.this) {
                    referenceTimelineThread = null;
                    FakeMonotonicClock.this.notifyAll();
                }
                closed = true;
            }
        }
    }

    private MonotonicTime currentTime;
    private Thread referenceTimelineThread;

    public FakeMonotonicClock() {
        this.currentTime = new MonotonicTime(0);
        this.referenceTimelineThread = null;
    }

    @Override
    public synchronized MonotonicTime getTime() {
        return currentTime;
    }

    @Override
    public synchronized void sleep(Duration duration) throws InterruptedException {
        if (!duration.isNegative() && !duration.isZero()) {
            MonotonicTime targetTime = currentTime.plus(duration);
            while (referenceTimelineThread != null
                    && !referenceTimelineThread.equals(Thread.currentThread())
                    && targetTime.isAfter(currentTime)
            ) {
                wait();
            }

            if (targetTime.isAfter(currentTime)) {
                currentTime = targetTime;
                notifyAll();
            }
        }
    }

    public synchronized ReferenceTimeline referenceTimeline() {
        if (referenceTimelineThread != null) {
            throw new IllegalStateException("Reference timeline thread is already set to: " + referenceTimelineThread);
        } else {
            referenceTimelineThread = Thread.currentThread();
            return new ReferenceTimeline();
        }
    }
}
