package ru.yandex.direct.utils;

import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nullable;

/**
 * Объект сотояния для сбора статистики по полю объекта, такой как:
 * <ul>
 * <li>максимум</li>
 * </ul>
 * <p>
 * Предназначен для использования со {@linkplain java.util.stream streams} объектов, для полей которых хочется собирать
 * статистику за один проход по стриму.
 * <p>
 * Значения {@code null} в сравнениях не участвуют.
 * <p>
 * Пример - возможность получить максимальный значение поля age за один проход с группировкой
 * по полю name класса Person:
 * <pre>{@code
 *  class Person {
 *      private Integer age;
 *      private String name;
 *
 *      public Person(Integer age, String name) {
 *          this.age = age;
 *          this.name = name;
 *      }
 *
 *      public Integer getAge() {
 *          return age;
 *      }
 *
 *      public String getName() {
 *          return name;
 *         }
 *  }
 *
 *  class MyClass {
 *      Person[] data = {new Person(20, "Nick"), new Person(null, "Dan"), new Person(25, "Zack")};
 *
 *      public static void main(String[] args) {
 *          FieldExtremums<Person, Integer> personAgeExtremums = new FieldExtremums<>(Person::getAge);
 *          Map<String, List<Person>> personsByName =
 *              Stream.of(data).peek(personAgeExtremums).collect(groupingBy(Person::getName));
 *          Integer maxAge = personAgeExtremums.getMax();
 *      }
 *  }
 * }</pre>
 *
 * @param <T> тип исходного объекта
 * @param <R> тип поля объекта (и результирующих значений статистики)
 * @implNote Реализация НЕ-потокобезопасна. Не следует использовать с параллельными стримами
 * @implNote Минимум пока не потребовался и поэтому не реализован
 */
public class FieldExtremums<T, R extends Comparable<? super R>> implements Consumer<T> {
    private R max = null;
    private final Function<T, R> fieldExtractor;

    /**
     * Конструктор.
     * Создает пустой объект, значение максимума равно {@code null}
     *
     * @param fieldExtractor метод для получения поля из объекта
     */
    public FieldExtremums(Function<T, R> fieldExtractor) {
        this.fieldExtractor = fieldExtractor;
    }

    /**
     * Записывает новые значения поля, получаемого из объекта методом {@code fieldExtractor}
     *
     * @param object входной объект
     */
    @Override
    public void accept(T object) {
        R value = fieldExtractor.apply(object);
        if (value != null) {
            if (max == null) {
                max = value;
            } else if (max.compareTo(value) < 0) {
                max = value;
            }
        }
    }

    /**
     * Возвращает записанное максимальное значение поля или {@code null},
     * если значений не было или все они были равны {@code null}
     *
     * @return максимальное значение или {@code null}
     */
    @Nullable
    public R getMax() {
        return max;
    }
}
