package ru.yandex.direct.utils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.StreamSupport.stream;

public class FunctionalUtils {
    private FunctionalUtils() {
    }

    /**
     * Преобразует {@link Iterable}, используя функцию
     *
     * @param source        - фильтруемый элементы
     * @param elementMapper - функция-преобразователь
     * @param <S>           - (Source type)тип исходных элементов
     * @param <D>           - (Destination type)тип преобразованных элементов
     * @return список преобразованных элементов из source; null, если source == null
     */
    public static <S, D> List<D> mapList(Iterable<? extends S> source, Function<? super S, ? extends D> elementMapper) {
        if (source == null) {
            return null;
        }
        return stream(source.spliterator(), false)
                .map(elementMapper)
                .collect(toList());
    }

    /**
     * Преобразует {@link Set}, используя функцию
     *
     * @param source        - фильтруемый элементы
     * @param elementMapper - функция-преобразователь
     * @param <S>           - (Source type)тип исходных элементов
     * @param <D>           - (Destination type)тип преобразованных элементов
     * @return список преобразованных элементов из source; null, если source == null
     */
    public static <S, D> Set<D> mapSet(Set<? extends S> source, Function<? super S, ? extends D> elementMapper) {
        return listToSet(source, elementMapper);
    }

    @SuppressWarnings("unchecked")
    public static <T, E> List<E> selectList(Iterable<T> source, Class<E> clazz) {
        return (List<E>) filterList(source, clazz::isInstance);
    }

    /**
     * Фильтрует {@link Iterable}, используя предикат
     *
     * @param source    - элементы для фильтрации
     * @param predicate - условие фильтрации
     * @param <T>       - тип фильтруемых элементов
     * @return список элементов из source, подходящих под условие; null, если source == null
     */
    public static <T> List<T> filterList(Iterable<? extends T> source, Predicate<? super T> predicate) {
        if (source == null) {
            return null;
        }
        return stream(source.spliterator(), false)
                .filter(predicate)
                .collect(toList());
    }

    /**
     * Фильтрует и преобразует {@link Iterable}, используя функцию и предикат
     *
     * @param source        - элементы для фильтрации
     * @param predicate     - условие фильтрации
     * @param elementMapper - функция-преобразователь
     * @param <T>           - тип фильтруемых элементов
     * @param <D>           - (Destination type)тип преобразованных элементов
     * @return список элементов из source, подходящих под условие; null, если source == null
     */
    public static <T, D> List<D> filterAndMapList(Iterable<? extends T> source, Predicate<? super T> predicate,
                                                  Function<? super T, ? extends D> elementMapper) {
        if (source == null) {
            return null;
        }
        return stream(source.spliterator(), false)
                .filter(predicate)
                .map(elementMapper)
                .collect(toList());
    }

    /**
     * Преобразует и фильтрует {@link Iterable}, используя предикат и функцию
     *
     * @param source        - элементы для преобразования и фильтрации
     * @param elementMapper - функция-преобразователь
     * @param predicate     - условие фильтрации
     * @param <T>           - (Source type)тип преобразуемых элементов
     * @param <D>           - (Destination type)тип фильтруемых элементов
     * @return список элементов из source, подходящих под условие; null, если source == null
     */
    public static <T, D> List<D> mapAndFilterList(Iterable<? extends T> source,
                                                  Function<? super T, ? extends D> elementMapper,
                                                  Predicate<? super D> predicate) {
        if (source == null) {
            return null;
        }
        return stream(source.spliterator(), false)
                .map(elementMapper)
                .filter(predicate)
                .collect(toList());
    }

    /**
     * Создание из списка мапы
     *
     * @param source    - список элементов для маппинга без дубликатов
     * @param keyMapper - маппер, получающий на вход значения, и отдающий ключи
     * @param <S>       - (Source type)тип элементов исходного списка
     * @param <D>       - (Destination type)тип элементов мапы
     * @param <K>       - (Key type)тип ключей мапы
     * @return мапа, где значения - элементы списка, а ключи - результат работы KeyMapper по значениям
     */
    public static <K, D, S extends D> Map<K, D> listToMap(
            Iterable<? extends S> source,
            Function<S, ? extends K> keyMapper) {
        return listToMap(source, keyMapper, identity());
    }

    /**
     * Создание из списка мапы
     *
     * @param source      - список элементов для маппинга без дубликатов
     * @param keyMapper   - маппер, получающий на вход значения, и отдающий ключи
     * @param valueMapper - маппер, получающий на вход значения итератора, и отдающий значения хеша
     * @param <S>         - (Source type)тип элементов исходного списка
     * @param <D>         - (Destination type)тип элементов мапы
     * @param <K>         - (Key type)тип ключей мапы
     * @param <V>         - (Value type)тип значений мапы
     * @return мапа, где значения - элементы списка, а ключи - результат работы KeyMapper по значениям
     */
    public static <K, V, D, S extends D> Map<K, V> listToMap(
            Iterable<? extends S> source,
            Function<S, ? extends K> keyMapper, Function<S, ? extends V> valueMapper) {
        if (source == null) {
            return null;
        }
        return stream(source.spliterator(), false)
                .collect(toMap(keyMapper, valueMapper));
    }

    /**
     * Создание из списка сета
     *
     * @param source    - исходный список элементов
     * @param <S>       - (Source type) тип элементов списка
     * @return сет с элементами из исходного списка
     */
    public static <S> Set<S> listToSet(Iterable<? extends S> source) {
        return listToSet(source, identity());
    }

    /**
     * Создание из списка сета
     *
     * @param source    - список элементов для маппинга
     * @param keyMapper - маппер, получающий на вход значения, и отдающий ключи
     * @param <S>       - (Source type) тип элементов исходного списка
     * @param <D>       - (Destination type)тип элементов результирующего множества
     * @return сет, где значения - результат работы KeyMapper по элементам списка
     */
    public static <D, S> Set<D> listToSet(
            Iterable<? extends S> source,
            Function<? super S, ? extends D> keyMapper) {
        if (source == null) {
            return null;
        }
        return stream(source.spliterator(), false)
                .map(keyMapper)
                .collect(toSet());
    }

    /**
     * Фильтрует {@link Iterable} и сладывает результат в {@link Set}, используя предикат
     *
     * @param source    - элементы для фильтрации
     * @param predicate - условие фильтрации
     * @param <T>       - тип фильтруемых элементов
     * @return список элементов из source, подходящих под условие; null, если source == null
     */
    public static <T> Set<T> filterToSet(Iterable<? extends T> source, Predicate<? super T> predicate) {
        return filterAndMapToSet(source, predicate, identity());
    }

    /**
     * Фильтрует и преобразует {@link Iterable} в {@link Set}, используя функцию и предикат
     *
     * @param source        - элементы для фильтрации
     * @param predicate     - условие фильтрации
     * @param elementMapper - функция-преобразователь
     * @param <T>           - тип фильтруемых элементов
     * @param <D>           - (Destination type)тип преобразованных элементов
     * @return список элементов из source, подходящих под условие; null, если source == null
     */
    public static <T, D> Set<D> filterAndMapToSet(Iterable<? extends T> source, Predicate<? super T> predicate,
                                                  Function<? super T, ? extends D> elementMapper) {
        if (source == null) {
            return null;
        }
        return stream(source.spliterator(), false)
                .filter(predicate)
                .map(elementMapper)
                .collect(toSet());
    }

    /**
     * Фильтрует и преобразует {@link Iterable} в {@link Set}, используя предикат и функцию
     *
     * @param source        - элементы для фильтрации
     * @param predicate     - условие фильтрации
     * @param elementMapper - функция-преобразователь
     * @param <T>           - тип фильтруемых элементов
     * @param <D>           - (Destination type)тип преобразованных элементов
     * @return список элементов из source, подходящих под условие; null, если source == null
     */
    public static <T, D> Set<D> mapAndFilterToSet(Iterable<? extends T> source,
                                                  Function<? super T, ? extends D> elementMapper,
                                                  Predicate<? super D> predicate) {
        if (source == null) {
            return null;
        }
        return stream(source.spliterator(), false)
                .map(elementMapper)
                .filter(predicate)
                .collect(toSet());
    }

    /**
     * Создание списка из списка списков
     *
     * @param container  Список, содержащий элементы-контейнеры
     * @param itemGetter Функция, получающая из элемента-контейнера список конечных элекментов
     * @param <C>        Тип элемента-контейнера
     * @param <I>        Тип конечного элемента
     * @return Единый список конечных элементов, полученных из элементов-контейнеров
     */
    public static <C, I> List<I> flatMap(Iterable<? extends C> container,
                                         Function<? super C, Iterable<? extends I>> itemGetter) {
        return stream(container.spliterator(), false)
                .flatMap(c -> stream(itemGetter.apply(c).spliterator(), false))
                .collect(toList());
    }

    /**
     * Создание списка из всех элементов всех коллекций из данной коллекции
     *
     * @param container Коллекция, содержащая коллекции элементов
     * @param <I>       Тип элементов
     * @return Список из всех элементов всех коллекций
     */
    public static <I> List<I> flatten(Iterable<? extends Iterable<? extends I>> container) {
        return flatMap(container, Function.identity());
    }

    /**
     * Создание Set'а из списка списков
     *
     * @param container  Список, содержащий элементы-контейнеры
     * @param itemGetter Функция, получающая из элемента-контейнера список конечных элекментов
     * @param <C>        Тип элемента-контейнера
     * @param <I>        Тип конечного элемента
     * @return {@link Set} элементов, полученных из элементов-контейнеров
     */
    public static <C, I> Set<I> flatMapToSet(Iterable<? extends C> container,
                                             Function<? super C, Iterable<? extends I>> itemGetter) {
        return stream(container.spliterator(), false)
                .flatMap(c -> stream(itemGetter.apply(c).spliterator(), false))
                .collect(toSet());
    }

    /**
     * Создание списка из списка списков с игнорированием null списков и возможностью выбора результирующей коллекции.
     *
     * @param container  Список, содержащий элементы-контейнеры
     * @param itemGetter Функция, получающая из элемента-контейнера список конечных элекментов
     * @param collector  Коллектор
     * @param <C>        Тип элемента-контейнера
     * @param <I>        Тип конечного элемента
     * @param <K>        Тип конечного контейнера
     * @return Единый список конечных элементов, полученных из элементов-контейнеров
     */
    public static <C, I, K> K nullSafetyFlatMap(
            Iterable<? extends C> container,
            Function<? super C, Iterable<? extends I>> itemGetter,
            Collector<I, ?, K> collector) {
        return stream(container.spliterator(), false)
                .map(itemGetter::apply)
                .filter(Objects::nonNull)
                .flatMap(c -> stream(c.spliterator(), false))
                .collect(collector);
    }

    public static List<Integer> intRange(int startInclusive, int endExclusive) {
        return IntStream.range(startInclusive, endExclusive)
                .boxed()
                .collect(toList());
    }

    public static <X> List<X> extractSubList(List<X> list, List<Integer> indexes) {
        return indexes.stream().map(list::get).collect(toList());
    }

    /**
     * Возвращает список индексов элементов, для которых выполняется предикат
     */
    public static <T> List<Integer> selectIndexes(List<T> items, Predicate<T> predicate) {
        return IntStream.range(0, items.size())
                .filter(idx -> predicate.test(items.get(idx)))
                .boxed()
                .collect(toList());
    }

    /**
     * Для списка размера size, возвращает список индексов, которые не указаны в indexesToExclude.
     *
     * К примеру для
     *      invertIndexes(3, List.of(1))
     * должно возратить
     *      List.of(0, 2)
     */
    public static List<Integer> invertIndexes(int size, Collection<Integer> indexesToExclude) {
        var indexesToExcludeSet = Set.copyOf(indexesToExclude);
        return IntStream.range(0, size)
                .boxed()
                .filter(idx -> !indexesToExcludeSet.contains(idx))
                .collect(toList());
    }

    /**
     * {@code batchProcessor} -- функция, которая каким-либо образом обрабатывает список элементов и возвращает
     * результат для каждого. Такие функции используются у нас в тех местах, где затратно обрабатывать поэлементно,
     * например, запросы в БД гораздо дешевле выполнить сразу для нескольких аргументов, нежели для каждого отдельно.
     * <p>
     * Смысл этой утилиты лишь проверить, что для каждого входящего элемента вернулся результат. От этого зависит
     * правильность работы других механизмов, а такую проверку писать везде накладно.
     * Для пустого списка, {@code batchProcessor} не вызывается
     */
    public static <I, R> List<R> batchApply(Function<List<I>, List<R>> batchProcessor, List<I> items) {
        if (items.isEmpty()) {
            return List.of();
        }
        List<R> result = batchProcessor.apply(items);
        checkState(result.size() == items.size(), "Batch processor must return result for each input value");
        return result;
    }

    /**
     * Выполнить разные обработку частей списка разными функциями. Функции firstBatchProcessor, secondBatchProcessor
     * выполняют обработку не поэлементно, а всего списка за раз (к примеру один запрос получения объектов из базы).
     * Результат склеивается из результатов firstBatchProcessor и secondBatchProcessor, так, чтобы он соотвествовал
     * элементам входящего списка.
     *
     * @param items                 входящий список элементов
     * @param shouldDispatchToFirst предикат, который применяется поэлементно на входящем списке, и говорит, что
     *                              элемент должен отправится на обработку в firstBatchProcessor
     * @param firstBatchProcessor   функция, обрабатывающая пачку входящих элементов
     * @param secondBatchProcessor  функция, обрабатывающая пачку входящих элементов
     * @param <I>                   тип элементов входящего списка
     * @param <R>                   тип элементов результирующего списка
     * @return список результатов
     */
    public static <I, R> List<R> batchDispatch(List<I> items,
                                               Predicate<I> shouldDispatchToFirst,
                                               Function<List<I>, List<R>> firstBatchProcessor,
                                               Function<List<I>, List<R>> secondBatchProcessor) {
        List<Integer> indexesToFirst = selectIndexes(items, shouldDispatchToFirst);
        return batchDispatch(items, indexesToFirst, firstBatchProcessor, secondBatchProcessor);
    }

    public static <I, R> List<R> batchDispatch(List<I> items,
                                               List<Integer> indexesToFirst,
                                               Function<List<I>, List<R>> firstBatchProcessor,
                                               Function<List<I>, List<R>> secondBatchProcessor) {
        List<R> resultFromFirst = batchApply(firstBatchProcessor, extractSubList(items, indexesToFirst));
        List<R> resultFromSecond = batchApply(
                secondBatchProcessor, extractSubList(items, invertIndexes(items.size(), indexesToFirst)));
        return mergeSubLists(resultFromFirst, resultFromSecond, indexesToFirst);
    }

    private static <R> List<R> mergeSubLists(List<R> firstList, List<R> secondList,
                                             List<Integer> firstListResultIndexes) {
        checkArgument(firstList.size() == firstListResultIndexes.size());

        Set<Integer> indexesForFirstList = Set.copyOf(firstListResultIndexes);
        int resultSize = firstList.size() + secondList.size();

        var firstIterator = firstList.iterator();
        var secondIterator = secondList.iterator();

        return IntStream.range(0, resultSize)
                .mapToObj(idx -> indexesForFirstList.contains(idx) ? firstIterator.next() : secondIterator.next())
                .collect(toList());
    }

    /**
     * @return {@link Supplier}, который всегда бросает {@link UnsupportedOperationException}.
     */
    public static <T> Supplier<T> unsupportedSupplier() {
        return () -> {
            throw new UnsupportedOperationException();
        };
    }

    /**
     * @return {@link Function}, которая всегда бросает {@link UnsupportedOperationException}.
     */
    public static <T, R> Function<T, R> unsupportedFunction() {
        return ignored -> {
            throw new UnsupportedOperationException();
        };
    }

    /**
     * Превращает мапу, значениями которой являются списки с элементами типа X,
     * в мапу, значениями которой являются списки с элементами типа T.
     * <p>
     * Пользоваться осторожно: приводимость не проверяется компилятором.
     *
     * @param map входная мапа
     * @param <X> тип элементов списков, являющихся значениями входной мапы
     * @param <T> тип элементов списков, являющихся значениями выходной мапы
     */
    @SuppressWarnings("unchecked")
    public static <X, T> Map<Long, List<T>> castListElementTypeOfMap(Map<Long, List<X>> map) {
        Map<Long, List<T>> res = new HashMap<>(map.size());

        for (Map.Entry<Long, List<X>> entry : map.entrySet()) {
            List<T> list = new ArrayList<>(entry.getValue().size());

            for (X x : entry.getValue()) {
                list.add((T) x);
            }

            res.put(entry.getKey(), list);
        }

        return res;
    }

    /**
     * Получить сет из двух коллекций любого типа
     * @param c1 первая коллекция
     * @param c2 вторая коллекция
     * @param <T> тип элемента коллекций
     */
    public static <T> Set<T> setUnion(Collection<T> c1, Collection<T> c2) {
        return Stream.of(c1, c2).flatMap(Collection::stream).collect(Collectors.toSet());
    }

    /**
     * Выполняет конкатенацию двух списков
     */
    public static <T> List<T> concat(List<T> c1, List<T> c2) {
        List<T> list = new ArrayList<>(c1.size() + c2.size());
        list.addAll(c1);
        list.addAll(c2);
        return list;
    }
}
