package ru.yandex.direct.utils;

import java.time.Duration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Позволяет программе аккуратно завершиться и освободить ресурсы, когда в jvm прилетает SIGINT, SIGTERM и прочее.
 * <p>
 * Работает очень просто: в случае jvm shutdown делается interrupt() на тот тред,
 * в котором был создан GracefulShutdownHook (обычно главный тред).
 * <p>
 * Если тред после interrupt-а не завершился за gracefulShutdownTimeout, вызывается Runtime.getRuntime().halt(1).
 */
public class GracefulShutdownHook implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(GracefulShutdownHook.class);

    private final Thread mainThread;
    private final Thread shutdownThread;
    private final Duration gracefulShutdownTimeout;
    private boolean shutdownInProgress;
    private boolean shutdownComplete;

    public GracefulShutdownHook(Duration gracefulShutdownTimeout) {
        this.gracefulShutdownTimeout = gracefulShutdownTimeout;
        shutdownInProgress = false;
        shutdownComplete = false;
        shutdownThread = new NamedThreadFactory("GracefulShutdownHook").newThread(this::doShutdown);
        mainThread = Thread.currentThread();
        Runtime.getRuntime().addShutdownHook(shutdownThread);
    }

    public synchronized boolean isShuttingDown() {
        return shutdownInProgress;
    }

    private synchronized boolean isShutdownComplete() {
        return shutdownComplete;
    }

    @Override
    public synchronized void close() {
        if (shutdownInProgress) {
            shutdownThread.interrupt();
        } else {
            Runtime.getRuntime().removeShutdownHook(shutdownThread);
        }
        shutdownComplete = true;
    }

    private void doShutdown() {
        if (!isShutdownComplete()) {
            synchronized (this) {
                shutdownInProgress = true;
                mainThread.interrupt();
            }
            logger.warn(
                    "Shutdown is in progress, waiting for graceful shutdown in " +
                            gracefulShutdownTimeout.toMillis() / 1000 + " seconds..."
            );
            try {
                try {
                    Thread.sleep(gracefulShutdownTimeout.toMillis());
                } catch (InterruptedException exc) {
                    Thread.currentThread().interrupt();
                }
            } finally {
                if (isShutdownComplete()) {
                    logger.warn("Graceful shutdown completed sucessfully.");
                } else {
                    logger.warn("Graceful shutdown timed out or interrupted, halting...");
                    Runtime.getRuntime().halt(1);
                }
            }
        }
    }
}
