package ru.yandex.direct.utils;

import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;

import org.apache.commons.codec.binary.Hex;
import org.apache.commons.lang3.StringUtils;

/**
 * Вспомогательные процедуры для encoding/encryption данных. Аналог - HashingTools.pm
 */
public class HashingUtils {
    public static final String MIME_DISALLOWED_CHARS = "+/";
    public static final String MIME_DISALLOWED_CHARS_REPLACEMENT = "-_";

    private HashingUtils() {
    }

    public static byte[] getMd5Hash(byte[] bytes) {
        return getMd5MessageDigest().digest(bytes);
    }

    public static BigInteger getMd5HashUtf8(String str) {
        return getMd5HashAsBigInteger(str.getBytes(StandardCharsets.UTF_8));
    }

    public static BigInteger getMd5HashAsBigInteger(byte[] bytes) {
        byte[] digest = getMd5Hash(bytes);
        return new BigInteger(1, digest);
    }

    public static String getMd5HashUtf8AsHexString(String str) {
        return getMd5HashAsHexString(str.getBytes(StandardCharsets.UTF_8));
    }

    public static String getMd5HashAsHexString(byte[] bytes) {
        byte[] digest = getMd5Hash(bytes);
        return new String(Hex.encodeHex(digest, true));
    }

    public static String getMd5HashAsBase64YaString(byte[] bytes) {
        byte[] digest = getMd5Hash(bytes);
        return encode64Ya(digest);
    }

    public static String getMd5HashAsBase64YaStringWithoutPadding(byte[] bytes) {
        byte[] digest = getMd5Hash(bytes);
        return encode64YaWithoutPadding(digest);
    }

    /**
     * Аналог получения половины md5-хэша из perl
     *
     * @param str строка, по которой требуется вычислить md5-хэш
     * @return половина md5-хэша
     */
    public static BigInteger getMd5HalfHashUtf8(String str) {
        return getMd5HalfHash(str.getBytes(StandardCharsets.UTF_8));
    }

    /**
     * Аналог получения половины md5-хэша из perl
     *
     * @param bytes массив байтов, по которому требуется вычислить md5-хэш
     * @return половина md5-хэша
     */
    public static BigInteger getMd5HalfHash(byte[] bytes) {
        final int halfLength = 8;
        byte[] digest = getMd5Hash(bytes);
        byte[] halfDigest = new byte[halfLength];
        System.arraycopy(digest, 0, halfDigest, 0, halfLength);
        return new BigInteger(1, halfDigest);
    }

    /**
     * Вычисление хеша по строке алгоритмом, используемым в БК
     * Аналог MD5::EndHalfMix из c++ получения
     *
     * @param digest массив байтов - md5-хэш
     * @return 8-byte xor-based mix
     * @implNote портированная версия БКшной реализации на c++
     * @see <a href=https://a.yandex-team.ru/arc/trunk/arcadia/library/cpp/digest/md5/md5.cpp?rev=r8654754#L219>
     * MD5::EndHalfMix</a>
     */
    public static BigInteger getMd5HalfMix(byte[] digest) {
        var halfMix = BigInteger.valueOf(0);
        for (int i = 3; i >= 0; i--) {
            halfMix = halfMix
                    .or(BigInteger.valueOf(Byte.toUnsignedInt(digest[i]) ^ Byte.toUnsignedInt(digest[8 + i]))
                            .shiftLeft(3 - i << 3));
            halfMix = halfMix
                    .or(BigInteger.valueOf(Byte.toUnsignedInt(digest[4 + i]) ^ Byte.toUnsignedInt(digest[12 + i]))
                            .shiftLeft(7 - i << 3));
        }
        return halfMix;
    }

    public static String getSha256Hash(String str) {
        byte[] digest = getSha256MessageDigest().digest(str.getBytes());
        return new String(Hex.encodeHex(digest, true));
    }

    public static String getSha512Hash(String str) {
        byte[] digest = getSha512MessageDigest().digest(str.getBytes());
        return new String(Hex.encodeHex(digest, true));
    }

    /**
     * CGI-friendly версия mime-кодера:
     * <ul>
     * <li>
     * Сначала производится обычное mime-кодирование
     * </li>
     * <li>
     * В получившейся последовательности символы из {@link #MIME_DISALLOWED_CHARS} меняются на
     * {@link #MIME_DISALLOWED_CHARS_REPLACEMENT}
     * </li>
     * </ul>
     *
     * @param src кодируемые данные
     * @return закодированные данные
     * @see Base64#getMimeEncoder()
     */
    public static String encode64Ya(byte[] src) {
        Base64.Encoder encoder = Base64.getMimeEncoder();
        String mimeEncoded = new String(encoder.encode(src), StandardCharsets.US_ASCII);
        return StringUtils.replaceChars(mimeEncoded, MIME_DISALLOWED_CHARS, MIME_DISALLOWED_CHARS_REPLACEMENT);
    }

    public static String encode64YaWithoutPadding(byte[] src) {
        return Base64.getUrlEncoder().withoutPadding().encodeToString(src);
    }

    /**
     * Парный декодер для {@link #encode64Ya(byte[])}
     *
     * @param src декодируемые данные
     * @return декодированные данные
     * @see Base64#getMimeDecoder()
     */
    public static byte[] decode64Ya(String src) throws IllegalArgumentException {
        String mimeEncoded = StringUtils.replaceChars(src, HashingUtils.MIME_DISALLOWED_CHARS_REPLACEMENT,
                HashingUtils.MIME_DISALLOWED_CHARS);
        Base64.Decoder mimeDecoder = Base64.getMimeDecoder();
        return mimeDecoder.decode(mimeEncoded);
    }

    /**
     * Вычисление хеша по строке алгоритмом, используемым в БК для вычисления DataMD5(Phrase)
     *
     * @param str строка, по которой требуется вычислить md5-хэш
     * @return {@link BigInteger}, значения которого соответствуют {@code unsigned long}
     * @implNote портированная версия БКшной реализации на Perl
     * @see
     * <a href=https://a.yandex-team.ru/arc/trunk/arcadia/yabs/basic_packages/yabs-base-modules/Yabs/Funcs.pm#L1408>
     * Yabs::Funcs::md5int</a>
     */
    public static BigInteger getYabsMd5HalfHashUtf8(String str) {
        byte[] md5 = getMd5Hash(str.getBytes(StandardCharsets.UTF_8));
        ByteBuffer buffer = ByteBuffer.wrap(md5);

        // четыре 32-х битных числа (беззнаковые int), составляющие 128 бит md5
        long[] parts = new long[4];
        for (int i = 0; i < parts.length; i++) {
            parts[i] = Integer.toUnsignedLong(buffer.getInt(i * Integer.BYTES));
        }

        BigInteger high = BigInteger.valueOf(parts[1] ^ parts[3]);
        BigInteger low = BigInteger.valueOf(parts[0] ^ parts[2]);

        return high.shiftLeft(Integer.SIZE).or(low);
    }

    private static MessageDigest getMd5MessageDigest() {
        return getMessageDigest("MD5");
    }

    private static MessageDigest getSha256MessageDigest() {
        return getMessageDigest("SHA-256");
    }

    private static MessageDigest getSha512MessageDigest() {
        return getMessageDigest("SHA-512");
    }

    private static MessageDigest getMessageDigest(String algorithm) {
        try {
            return MessageDigest.getInstance(algorithm);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("can not retrieve java.security.MessageDigest for MD5 algorithm", e);
        }
    }
}
