package ru.yandex.direct.utils;

import java.io.IOException;
import java.util.Arrays;
import java.util.Map;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.deser.std.StdScalarDeserializer;
import com.google.common.collect.Maps;

/**
 * Десериализатор {@link Enum}'ов, не учитывающий регистра констант. Не умеет работать с номерами констант -
 * в этом случае десериализатор будет выбран в {@link JsonParser#readValueAs(Class)}.
 *
 * @param <T>
 */
public class IgnoreCaseEnumDeserializer<T extends Enum> extends StdScalarDeserializer<T> {
    private final Map<String, T> upperCaseNamesMap;

    public IgnoreCaseEnumDeserializer(Class<T> vc) {
        super(vc);
        upperCaseNamesMap = Maps.uniqueIndex(Arrays.asList(vc.getEnumConstants()),
                enumEl -> enumEl.name().toUpperCase());
    }

    @Override
    @SuppressWarnings("unchecked")
    public T deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
        JsonToken curr = p.getCurrentToken();
        if (curr == JsonToken.VALUE_STRING || curr == JsonToken.FIELD_NAME) {
            String upperCaseName = p.getText().toUpperCase();
            if (upperCaseNamesMap.containsKey(upperCaseName)) {
                return upperCaseNamesMap.get(upperCaseName);
            }
        }
        return p.readValueAs((Class<T>) handledType());
    }
}
