package ru.yandex.direct.utils;

import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * Функции для обработки InterruptedException
 */
public class Interrupts {
    private Interrupts() {
    }

    @FunctionalInterface
    public interface InterruptibleCheckedRunnable<E extends Exception> {
        void run() throws InterruptedException, E;
    }

    @FunctionalInterface
    public interface InterruptibleCheckedSupplier<T, E extends Exception> {
        T get() throws InterruptedException, E;
    }

    @FunctionalInterface
    public interface InterruptibleSupplier<T> {
        T get() throws InterruptedException;
    }

    @FunctionalInterface
    public interface InterruptibleFunction<T, R> {
        R apply(T value) throws InterruptedException;
    }

    @FunctionalInterface
    public interface InterruptibleConsumer<T> {
        void accept(T value) throws InterruptedException;
    }

    @FunctionalInterface
    public interface InterruptibleBiConsumer<T, U> {
        void accept(T t, U u) throws InterruptedException;
    }

    public static class CriticalTimeoutExpired extends RuntimeException {
        CriticalTimeoutExpired() {
        }
    }

    @FunctionalInterface
    public interface WaitDuration {
        boolean await(Duration timeout) throws InterruptedException;
    }

    @FunctionalInterface
    public interface WaitTimeUnit {
        boolean await(long timeout, TimeUnit unit) throws InterruptedException;
    }

    @FunctionalInterface
    public interface WaitMillisNanos {
        void await(long millis, int nanos) throws InterruptedException;
    }

    public static WaitDuration waitMillisNanos(WaitMillisNanos waitMethod) {
        return duration -> {
            waitMethod.await(duration.toMillis(), duration.getNano() % 1_000_000);
            return false;
        };
    }

    public static WaitDuration waitTimeUnit(WaitTimeUnit waitMethod) {
        return duration -> waitMethod.await(Math.max(1, duration.toMillis()), TimeUnit.MILLISECONDS);
    }

    /**
     * Вызывает waitMethod с параметром timeout. Каждый раз, когда waitMethod бросает InterruptedException,
     * waitMethod перезапускается с оставшимся таймаутом. Если таймаут заканчивается до того как waitMethod
     * возвращает true, выбрасывается исключение CriticalTimeoutExpired.
     *
     * @param timeout    - время ожидания
     * @param waitMethod - функция ожидания
     */
    public static void criticalTimeoutWait(Duration timeout, WaitMillisNanos waitMethod) {
        criticalTimeoutWait(timeout, waitMillisNanos(waitMethod), new CriticalTimeoutExpired());
    }

    public static void criticalTimeoutWait(Duration timeout, WaitTimeUnit waitMethod) {
        criticalTimeoutWait(timeout, waitTimeUnit(waitMethod), new CriticalTimeoutExpired());
    }

    public static void criticalTimeoutWait(Duration timeout, WaitDuration waitMethod) {
        criticalTimeoutWait(timeout, waitMethod, new CriticalTimeoutExpired());
    }

    public static void criticalTimeoutWait(Duration timeout, WaitDuration waitMethod,
                                           RuntimeException timeoutExpiredException) {
        if (!waitUninterruptibly(timeout, waitMethod)) {
            throw Objects.requireNonNull(timeoutExpiredException, "timeoutExpiredException must not be null");
        }
    }

    public static boolean waitUninterruptibly(Duration timeout, WaitDuration waitMethod) {
        return waitUninterruptibly(timeout, waitMethod, NanoTimeClock.CLOCK);
    }

    // S2142 - осознано не перевыставляем isInterrupted сразу
    // S134 - ругается на большую вложенность из-за if_(success)
    @SuppressWarnings({"squid:S2142", "squid:S134"})
    public static boolean waitUninterruptibly(Duration timeout, WaitDuration waitMethod, MonotonicClock clock) {
        boolean interrupted = false;
        try {
            MonotonicTime now = clock.getTime();
            MonotonicTime deadline = now.plus(timeout);
            while (now.isBefore(deadline)) {
                try {
                    boolean success = waitMethod.await(deadline.minus(now));
                    if (success) {
                        return true;
                    }
                } catch (InterruptedException exc) {
                    interrupted = true;
                }
                now = clock.getTime();
            }
        } finally {
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
        return false;
    }

    /**
     * То же что и failingRun только с возвращаемым значением
     */
    public static <T, E extends Exception> T failingGet(InterruptibleCheckedSupplier<T, E> supplier) throws E {
        try {
            return supplier.get();
        } catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(exc);
        }
    }

    /**
     * То же что и silentRun только с возвращаемым значением
     * <p>
     * Использовать с крайней осторожностью!
     * Проглатывание InterruptedException может привести к зависанию программы!
     * <p>
     * Хорошим тоном будет оставить комментарий в месте вызова этого метода с объяснением,
     * почему такое поведение оправдано.
     */
    public static <T, E extends Exception> T silentGet(InterruptibleCheckedSupplier<T, E> supplier, T defaultResult)
            throws E {
        try {
            return supplier.get();
        } catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            return defaultResult;
        }
    }

    /**
     * Выполняет runnable. Завершается с исключением RuntimeInterruptedException если runnable бросает
     * InterruptedException.
     */
    public static <T extends Exception> void failingRun(InterruptibleCheckedRunnable<T> runnable) throws T {
        try {
            runnable.run();
        } catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(exc);
        }
    }

    /**
     * Выполняет runnable. Если runnable завершается с InterruptedException, то silentRun прекращается,
     * исключение проглатывается и у треда снова выставляется флажок interrupted.
     * <p>
     * Использовать с крайней осторожностью!
     * Проглатывание InterruptedException может привести к зависанию программы!
     * <p>
     * Хорошим тоном будет оставить комментарий в месте вызова этого метода с объяснением,
     * почему такое поведение оправдано.
     */
    public static <T extends Exception> void silentRun(InterruptibleCheckedRunnable<T> runnable) throws T {
        try {
            runnable.run();
        } catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
        }
    }

    /**
     * Выполняет runnable. Перезапускает его каждый раз при получении InterruptedException.
     * Если в процессе выполнения хотя бы раз случился InterruptedException, то после успешного
     * завершения runnable будет брошено исключение RuntimeInterruptedException.
     * <p>
     * Использовать с осторожностью!
     * Во-первых, runnable должен представлять собой идемпотентную операцию,
     * во-вторых, нужно быть абсолютно уверенным в том, что runnable рано или поздно завершится,
     * иначе можно получить зависшую программу.
     * <p>
     * Хорошим тоном будет оставить комментарий в месте вызова этого метода с объяснением,
     * почему такое поведение оправдано.
     */
    public static <T extends Exception> void resurrectingRun(InterruptibleCheckedRunnable<T> runnable) throws T {
        boolean interrupted = false;
        try {
            while (true) {
                try {
                    runnable.run();
                    break;
                } catch (InterruptedException exc) {
                    interrupted = true;
                }
            }
        } finally {
            if (interrupted) {
                Thread.currentThread().interrupt();
            }
        }
    }
}
