package ru.yandex.direct.utils;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.StringJoiner;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonLocation;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.util.JsonFormat;

import ru.yandex.direct.utils.json.BigDecimalSerializer;
import ru.yandex.direct.utils.json.LocalDateDeserializer;
import ru.yandex.direct.utils.json.LocalDateSerializer;
import ru.yandex.direct.utils.json.LocalDateTimeDeserializer;
import ru.yandex.direct.utils.json.LocalDateTimeSerializer;

public class JsonUtils {
    private static final String DESERIALIZATION_ERROR = "can not deserialize object from json";

    public static final ObjectMapper MAPPER = new ObjectMapper()
            .registerModule(new JavaTimeModule())
            .registerModule(new KotlinModule())
            .enable(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_USING_DEFAULT_VALUE)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    public static final ObjectMapper NONNULL_FIELDS_MAPPER = MAPPER.copy()
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    private static final ObjectMapper DETERMINISTIC_MAPPER = MAPPER.copy()
            .enable(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS);

    public static final JsonFormat.Printer PROTO_PRINTER = JsonFormat.printer();

    private JsonUtils() {
        // No instantiating, please
    }

    public static ObjectMapper getObjectMapper() {
        return MAPPER;
    }

    public static TypeFactory getTypeFactory() {
        return MAPPER.getTypeFactory();
    }

    /**
     * Десериализует объект из JSON
     *
     * @param json строка JSON
     * @param type тип объекта
     * @param <T>  тип объекта
     * @return десериализованный объект
     */
    public static <T> T fromJson(String json, Class<T> type) {
        try {
            return MAPPER.readValue(json, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(byte[] json, Class<T> type) {
        try {
            return MAPPER.readValue(json, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(byte[] json, JavaType valueType) {
        try {
            return MAPPER.readValue(json, valueType);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(InputStream inputStream, Class<T> type) {
        try {
            return MAPPER.readValue(inputStream, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(String json, JavaType type) {
        try {
            return MAPPER.readValue(json, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(String json, TypeReference<T> typeReference) {
        try {
            return MAPPER.readValue(json, typeReference);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromJson(JsonNode jsonNode, Class<T> type) {
        try {
            return MAPPER.treeToValue(jsonNode, type);
        } catch (IOException e) {
            throw new IllegalArgumentException("can not parse json", e);
        }
    }

    public static JsonNode fromJson(String json) {
        try {
            return MAPPER.readTree(json);
        } catch (IOException e) {
            throw new IllegalArgumentException("can not parse json", e);
        }
    }

    public static Module createLocalDateTimeModule() {
        SimpleModule module = new SimpleModule("LocalDateTimeModule");
        module.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer());
        module.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer());
        return module;
    }

    public static Module createLocalDateModule() {
        return new SimpleModule("LocalDateModule")
                .addSerializer(LocalDate.class, new LocalDateSerializer())
                .addDeserializer(LocalDate.class, new LocalDateDeserializer());
    }

    public static Module createBigDecimalModule() {
        return new SimpleModule("BigDecimal")
                .addSerializer(BigDecimal.class, new BigDecimalSerializer());
    }

    /**
     * Сериализует объект в JSON
     *
     * @param obj объект
     * @return строка JSON
     */
    public static String toJson(Object obj) {
        return toJson(obj, MAPPER);
    }

    public static String toJsonIgnoringNullFields(Object obj) {
        return toJson(obj, NONNULL_FIELDS_MAPPER);
    }

    private static String toJson(Object obj, ObjectMapper mapper) {
        try {
            if (obj instanceof MessageOrBuilder) {
                return PROTO_PRINTER.print((MessageOrBuilder) obj);
            } else {
                return mapper.writeValueAsString(obj);
            }
        } catch (JsonProcessingException | InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    /**
     * Сериализует объект в JSON
     *
     * @param obj           объект
     * @param typeReference тип объекта
     * @return строка JSON
     */
    public static <T> String toJson(Object obj, TypeReference<T> typeReference) {
        try {
            return MAPPER.writer()
                    .forType(typeReference)
                    .writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to typed json", e);
        }
    }

    public static byte[] toJsonBytes(Object obj) {
        try {
            return MAPPER.writeValueAsBytes(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    public static byte[] toJsonBytesForType(JavaType type, Object obj) {
        try {
            return MAPPER.writerFor(type).writeValueAsBytes(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    @Nullable
    public static String toJsonNullable(Object obj) {
        return obj != null ? toJson(obj) : null;
    }

    /**
     * Преобразует коллекцию в JSON массив. null остается null, пустая коллекция также преобразуется в null
     * Если пустая коллекция не дожна преобразовываться в null, то можно использовать {@link #toJsonNullable(Object)}
     *
     * @param collection колекция для конвертирования в json
     * @return json-строка или null
     */
    @Nullable
    public static String toJsonCollectionEmptyToNull(@Nullable Collection<?> collection) {
        return collection == null || collection.isEmpty() ? null : toJson(collection);
    }

    /**
     * Преобразует объект в JSON строку в повторяемом виде.
     * Это не такой канонический вид, к которому мы привыкли в перле, т.к. ключи объекта выводятся в том порядке,
     * в котором они определены в классе.
     * Но хотя бы сортируются ключи хеш мапы, что делает результирующий JSON хотя бы предсказуемым.про
     */
    public static String toDeterministicJson(Object obj) {
        try {
            return DETERMINISTIC_MAPPER.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    /**
     * Извлекает координаты ошибки из JsonMappingException и преобразует в human-readable сообщение.
     * {@link Throwable#getMessage()} при этом игнорируется.
     */
    public static String extractJsonMappingErrorLocation(JsonMappingException ex) {
        List<JsonMappingException.Reference> referenceList = ex.getPath();
        StringJoiner joiner = new StringJoiner(" -> ", "[", "]");
        for (JsonMappingException.Reference ref : referenceList) {
            String fieldName = ref.getFieldName();
            if (fieldName != null) {
                joiner.add(fieldName);
            } else {
                final Object srcObject = ref.getFrom();
                if (srcObject instanceof List) {
                    joiner.add(String.format("array[%d]", ((List) srcObject).size()));
                } else {
                    if (srcObject instanceof Map) {
                        joiner.add("object");
                    }
                }
            }
        }
        JsonLocation location = ex.getLocation();
        return String.format("can't map json into internal data model; path: %s, line: %d, column: %d",
                joiner.toString(), location.getLineNr(), location.getColumnNr());
    }

    public static <T> T ifNotNull(JsonNode jsonNode, Function<JsonNode, T> converter) {
        if (jsonNode == null) {
            return null;
        }

        return jsonNode.isNull() ? null : converter.apply(jsonNode);
    }

    /**
     * Десериализует объект из JSON. Возвращает {@code null} если передана невалидная json-строка
     */
    public static <T> T fromJsonIgnoringErrors(String json, Class<T> type) {
        if (json == null) {
            return null;
        }
        try {
            return MAPPER.readValue(json, type);
        } catch (IOException e) {
            return null;
        }
    }
}
