package ru.yandex.direct.utils;

import java.io.IOException;
import java.io.InputStream;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.databind.SerializationFeature;

public class JsonUtilsWithMapper {
    public static final JsonUtilsWithMapper JSON_UTILS_WITH_HUMAN_READABLE_DATES = new JsonUtilsWithMapper(
            JsonUtils.MAPPER.copy().disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS));
    private static final String DESERIALIZATION_ERROR = "can not deserialize object from json";

    private final ObjectMapper mapper;
    private final ObjectWriter writer;

    public JsonUtilsWithMapper(ObjectMapper mapper) {
        this.mapper = mapper;
        writer = mapper.writer().with(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS);
    }

    /**
     * Десериализует объект из JSON
     *
     * @param json строка JSON
     * @param type тип объекта
     * @param <T>  тип объекта
     * @return десериализованный объект
     */
    public <T> T fromJson(String json, Class<T> type) {
        try {
            return mapper.readValue(json, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public <T> T fromJson(InputStream inputStream, Class<T> type) {
        try {
            return mapper.readValue(inputStream, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    /**
     * Преобразует объект в JSON строку в повторяемом виде.
     */
    public String toJson(Object obj) {
        try {
            if (obj == null) {
                return null;
            }
            return writer.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

}


