package ru.yandex.direct.utils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.StreamSupport.stream;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class ListUtils {
    private ListUtils() {
    }

    /**
     * Уменьшает размер списка, производя равномерный ресемплинг - оставляет первый и последний элементы, остальные
     * элементы между ними через равные промежутки
     *
     * @param source        список, который надо уменьшить
     * @param countOfPoints количество элементов в результирующем списке
     * @param <T>           тип элементов списка
     * @return список из countOfPoints элементов, построенный из исходного списка
     */
    public static <T> List<T> resample(List<T> source, int countOfPoints) {
        if (source.size() <= countOfPoints) {
            return source;
        }

        int countOfIntervals = countOfPoints - 1;
        int intervalWidthInFractions = source.size() - 1;

        return IntStream.range(0, countOfPoints)
                .mapToObj(i -> source.get(i * intervalWidthInFractions / countOfIntervals))
                .collect(toList());
    }

    /**
     * Возвращает элементы списка, которые есть в нём больше одного раза.
     *
     * @param list список, который сам не null и в котором нет элементов null
     * @return список дублирующихся элементов. В этом списке нет повторов. Элементы идут в порядке,
     * в котором они во второй раз встретились в исходном списке.
     */
    public static <T> List<T> findDuplicates(List<T> list) {
        if (list.isEmpty()) {
            return emptyList();
        }

        Set<T> occurredElements = new HashSet<>();
        Set<T> duplicatedElements = new LinkedHashSet<>();
        for (T element : list) {
            checkNotNull(element);

            if (!occurredElements.add(element)) {
                duplicatedElements.add(element);
            }
        }

        if (duplicatedElements.isEmpty()) {
            return emptyList();
        }

        return new ArrayList<>(duplicatedElements);
    }

    /**
     * Обрабатывает одним разом все уникальные элементы из исходного списка и возвращает новый список,
     * в котором каждый элемент - результат обработки элемента из исходного списка на той же позиции.
     * <p>
     * Пример:
     * <p>
     * <code>{@literal
     * List<Integer> source = Arrays.asList(0, 1, 0, 2, 1, 3);
     * List<Integer> result = batchTransform(
     * source,
     * 999,
     * numbers -> numbers.stream()
     * .filter(n -> n != 0)
     * .collect(Collectors.toMap(n -> n, n -> n * n)));
     * assert result.equals(Arrays.asList(999, 1, 999, 4, 1, 9));
     * }</code>
     *
     * @param list         Список исходных объектов.
     * @param defaultValue Значение, которое будет подставлено в результат, если в результате обработки не будет
     *                     объекта, соответствующего исходному объекту.
     * @param batchFetcher Функция, которая принимает на вход уникальные элементы из исходного списка и
     *                     возвращает ассоциативный массив из исходных объектов в результирующие.
     * @param <T>          Тип исходного объекта.
     * @param <R>          Тип результирующего объекта.
     * @return Список, количество элементов в котором будет равно количеству элементов в list.
     */
    public static <T, R> List<R> batchTransform(
            List<T> list,
            R defaultValue,
            Function<Collection<T>, Map<T, R>> batchFetcher) {
        Map<T, Collection<Integer>> positions = new HashMap<>();
        int pos = 0;
        for (T arg : list) {
            positions.computeIfAbsent(arg, k -> new ArrayList<>()).add(pos++);
        }
        Map<T, R> intermediateResultList = batchFetcher.apply(positions.keySet());
        List<R> resultList = new ArrayList<>(Collections.nCopies(list.size(), defaultValue));
        for (Map.Entry<T, R> result : intermediateResultList.entrySet()) {
            for (int pos2 : positions.getOrDefault(result.getKey(), emptyList())) {
                resultList.set(pos2, result.getValue());
            }
        }
        return resultList;
    }


    /**
     * Преобразует {@link Iterable} в список, состоящий из уникальных значений
     */
    @Nonnull
    public static <T> List<T> uniqueList(@Nonnull Iterable<T> source) {
        return stream(source.spliterator(), false)
                .distinct()
                .collect(toList());
    }

    /**
     * Преобразует {@link Iterable<Long>} в список c типом {@link Integer}
     * Делается проверка {@link Math#toIntExact}
     */
    @Nonnull
    public static List<Integer> longToIntegerList(@Nonnull Iterable<Long> source) {
        return mapList(source, Math::toIntExact);
    }

    /**
     * Преобразует {@link Iterable<Long>} в множество c типом {@link Integer}
     * Делается проверка {@link Math#toIntExact}
     */
    @Nonnull
    public static Set<Integer> longToIntegerSet(@Nonnull Iterable<Long> source) {
        return listToSet(source, Math::toIntExact);
    }

    /**
     * Преобразует {@link Iterable<Integer>} в список c типом {@link Long}
     */
    @Nonnull
    public static List<Long> integerToLongList(@Nonnull Iterable<Integer> source) {
        return mapList(source, Integer::longValue);
    }
}
