package ru.yandex.direct.utils;

import java.time.Duration;

import javax.annotation.Nonnull;

/**
 * Абстрактное монотонное время, возвращаемое MonotonicClock.getTime().
 * <p>
 * Монотонное означает, что вызов MonotonicClock.getTime() не может вернуть время меньше, чем предыдущий вызов.
 * <p>
 * Абстрактное, в том смысле, что абсолютное значение времени не несет никакого смысла.
 * Но есть вполне конкретный смысл в разности двух MonotonicTime:
 * <p>
 * MonotonicTime t1 = clock.getTime()
 * doSomething()
 * MonotonicTime t2 = clock.getTime()
 * System.out.println(t2.minus(t1).toNanos())
 * <p>
 * напечатает время в наносекундах, потраченное на вызов doSomething().
 */
public class MonotonicTime implements Comparable<MonotonicTime> {
    private long nanoTime;

    /**
     * Конструктор умышленно не публичный. Получать инстанс этого класса нужно через MonotonicClock.getTime().
     */
    MonotonicTime(long nanoTime) {
        this.nanoTime = nanoTime;
    }

    public MonotonicTime plus(Duration duration) {
        long newNanoTime = nanoTime + duration.toNanos();
        if (duration.isNegative()) {
            if (newNanoTime >= nanoTime) {
                throw new IllegalStateException(
                        "Monotonic time overflow: " + nanoTime + " - " + (-duration.toNanos()) + " = " + newNanoTime
                );
            }
        } else {
            if (newNanoTime < nanoTime) {
                throw new IllegalStateException(
                        "Monotonic time overflow: " + nanoTime + " + " + duration.toNanos() + " = " + newNanoTime
                );
            }
        }
        return new MonotonicTime(newNanoTime);
    }

    public MonotonicTime minus(Duration duration) {
        return plus(duration.negated());
    }

    public Duration minus(MonotonicTime other) {
        return Duration.ofNanos(nanoTime - other.nanoTime);
    }

    public boolean isAfter(MonotonicTime other) {
        return nanoTime > other.nanoTime;
    }

    public boolean isAtOrAfter(MonotonicTime other) {
        return nanoTime >= other.nanoTime;
    }

    public boolean isBefore(MonotonicTime other) {
        return this.nanoTime < other.nanoTime;
    }

    public boolean isAtOrBefore(MonotonicTime other) {
        return this.nanoTime <= other.nanoTime;
    }

    @Override
    public String toString() {
        return "MonotonicTime{" + nanoTime + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        MonotonicTime time = (MonotonicTime) o;

        return nanoTime == time.nanoTime;
    }

    @Override
    public int hashCode() {
        return (int) (nanoTime ^ (nanoTime >>> 32));
    }

    @Override
    public int compareTo(@Nonnull MonotonicTime other) {
        return Long.compare(this.nanoTime, other.nanoTime);
    }
}
