package ru.yandex.direct.utils;

import java.time.Duration;

/**
 * Обертка над System.nanoTime.
 * <p>
 * Появилась в качестве возможности мокать {@code System.nanoTime()} и {@code Thread.sleep()} в тестах.
 * В тестах вместо {@code NanoTimeClock} можно использовать {@code FakeMonotonicClock}.
 * <p>
 * Дополнительная плюшка - типизированное время (MonotonicTime), вместо long-а у nanoTime().
 * С long-ом проблема в том, что он не привязан к конкретным единицам измерения и при передаче
 * его между методами есть шанс запутаться и интерпретировать его в неправильных единицах, например,
 * в широкоиспользуемых миллисекундах.
 * <p>
 * Кроме того, в MonotonicTime есть контроль переполнения.
 */
public class NanoTimeClock implements MonotonicClock {
    public static final NanoTimeClock CLOCK = new NanoTimeClock();

    public static MonotonicTime now() {
        return CLOCK.getTime();
    }

    @Override
    public MonotonicTime getTime() {
        return new MonotonicTime(System.nanoTime());
    }

    @Override
    public void sleep(Duration duration) throws InterruptedException {
        if (!duration.isNegative() && !duration.isZero()) {
            Interrupts.waitMillisNanos(Thread::sleep).await(duration);
        }
    }
}
