package ru.yandex.direct.utils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.ObjectUtils;

@ParametersAreNonnullByDefault
public class NumberUtils {
    private NumberUtils() {
    }

    public static Long nullIfZero(@Nullable Long value) {
        return value != null && !value.equals(0L) ? value : null;
    }

    public static Long ifNullOrZero(@Nullable Long value, Long defaultValue) {
        return value != null && !value.equals(0L) ? value : defaultValue;
    }

    /**
     * Вычислить значение из числа, если оно не равно {@code null} и не равно {@code 0}.
     */
    public static <T> T ifNotZero(@Nullable Long value, Function<Long, T> converter) {
        return value != null && !value.equals(0L) ? converter.apply(value) : null;
    }

    /**
     * Вычислить значение из числа, если оно не равно {@code null} и не равно {@code 0}.
     */
    public static <T> T ifNotZero(@Nullable Double value, Function<Double, T> converter) {
        return value != null && !value.equals(0.0) ? converter.apply(value) : null;
    }

    /**
     * @return {@code true}, если величина больше нуля. Иначе {@code false}
     */
    public static boolean greaterThanZero(@Nullable BigDecimal value) {
        return value != null && value.compareTo(BigDecimal.ZERO) > 0;
    }

    public static boolean isZero(@Nullable BigDecimal value) {
        return value != null && value.compareTo(BigDecimal.ZERO) == 0;
    }

    /**
     * @return {@code true}, если значение равно {@code null} или нулю. Иначе {@code false}
     */
    public static boolean isNullOrZero(@Nullable BigDecimal value) {
        return value == null || value.compareTo(BigDecimal.ZERO) == 0;
    }

    public static boolean equalsByCompareTo(@Nullable BigDecimal a, @Nullable BigDecimal b) {
        return ObjectUtils.compare(a, b) == 0;
    }

    /**
     * Округляет вниз с заданной точностью.
     * Если {@code val} бесконечен или равен {@code NaN}, то значение возвращается как есть, без изменений
     *
     * @param val   значение для округления
     * @param scale нужное количество цифр после запятой
     * @return округленное значение
     */

    public static double roundDown(double val, int scale) {
        return round(val, scale, RoundingMode.DOWN);
    }

    /**
     * Копия с {@code org.apache.commons.math3.util.Precision#round(double, int, int)}}
     */
    public static double round(double val, int scale, RoundingMode roundingMode) {
        try {
            double rounded = new BigDecimal(Double.toString(val))
                    .setScale(scale, roundingMode)
                    .doubleValue();
            return rounded == 0d ? 0d * val : rounded;
        } catch (NumberFormatException ex) {
            if (Double.isInfinite(val)) {
                return val;
            } else {
                return Double.NaN;
            }
        }
    }
}
