package ru.yandex.direct.utils;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.base.Utf8;
import org.jetbrains.annotations.NotNull;

import static java.util.Arrays.stream;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

public class StringUtils {
    private StringUtils() {
    }

    public static String nullIfBlank(String str) {
        return org.apache.commons.lang3.StringUtils.defaultIfBlank(str, null);
    }

    /**
     * Вычислить значение из строки, если она не равна {@code null}, не пустая
     * и не состоит из пробельных символов (согласно apache StringUtils.isBlank).
     */
    public static <T> T ifNotBlank(String str, Function<String, T> converter) {
        return !org.apache.commons.lang3.StringUtils.isBlank(str) ? converter.apply(str) : null;
    }

    public static String replaceIgnoreCase(String source, String find, String replace) {
        return source.replaceAll("(?i)" + Pattern.quote(find), replace);
    }

    /**
     * Подсчитывает кол-во десятичных цифр ('0'..'9') в строке
     */
    public static int countDigits(String str) {
        if (str == null) {
            return 0;
        }
        int result = 0;
        for (int i = 0; i < str.length(); i++) {
            if ('0' <= str.charAt(i) && str.charAt(i) <= '9') {
                result++;
            }
        }
        return result;
    }

    /**
     * Быстрый разбор строки текста вида "123:567" (два неотрицательных числа, между ними разделитель) в два числа.
     * Работает в ~10 раз быстрее стандартных средств, встречается достаточно часто, поэтому отдельный код имеет смысл
     *
     * @param source         строка вида "число:число" (: - произвольный символ-разделитель)
     * @param separator      символ-разделитель
     * @param processResults функция, обрабатывающая полученные числа, и возвращающая контейнер с ними
     *                       (чтобы не всегда боксить)
     * @param <T>            тип результата-контейнера для полученных чисел
     * @return контейнер с распарсенными числами, полученный из фукнции-обработчика
     */
    public static <T> T fastParsePairOfUnsignedLongs(String source, char separator,
                                                     LongBiFunction<T> processResults) {
        int col = source.indexOf(separator);
        return processResults.apply(
                fastParseUnsignedLong(source, 0, col),
                fastParseUnsignedLong(source, col + 1, source.length()));
    }

    /**
     * Разбирает строку в неотрицательное целое число
     *
     * @param source строка, в ней должны содержаться только цифры, больше никаких символов
     * @return число (int), полученное из строки
     */
    public static int fastParseUnsignedInt(String source) {
        return (int) fastParseUnsignedLong(source, 0, source.length());
    }

    /**
     * Разбирает строку в неотрицательное целое число
     *
     * @param source строка, в ней должны содержаться только цифры, больше никаких символов
     * @return число (long), полученное из строки
     */
    public static long fastParseUnsignedLong(String source) {
        return fastParseUnsignedLong(source, 0, source.length());
    }

    /**
     * Разбирает строку в неотрицательное целое число
     *
     * @param source строка, в ней должны содержаться только цифры, больше никаких символов
     * @param begin  индекс начального символа (включая)
     * @param end    индекс конечного символа (исключая)
     * @return число (long), полученное из строки
     */
    public static long fastParseUnsignedLong(String source, int begin, int end) {
        if (begin == end) {
            throw new NumberFormatException("Can't parse unsigned long from string of zero length");
        }
        long cur = 0;
        for (int i = begin; i < end; i++) {
            char c = source.charAt(i);
            if (c >= '0' && c <= '9') {
                cur = cur * 10 + c - '0';
            } else {
                throw new NumberFormatException(String.format(
                        "Can't parse unsigned long from string \"%s\". Unexpected symbol '%c' on position %d",
                        source.substring(begin, end), c, i));
            }
        }
        return cur;
    }

    /**
     * Обрезает строку до определенной длины в байтах
     *
     * @param s      строка, которую надо обрезать
     * @param length длина в байтах, до которой будет обрезана строка
     * @return обрезанная строка до длины не более указанной (может быть менее, если последний двухбайтовый
     * символ будет разрезан пополам)
     */
    public static String cutUtf8ToLength(String s, int length) {
        if (Utf8.encodedLength(s) <= length) {
            return s;
        }
        ByteBuffer bb = ByteBuffer.wrap(new byte[length]);
        CharsetEncoder encoder = StandardCharsets.UTF_8.newEncoder();
        CharBuffer cb = CharBuffer.wrap(s);
        encoder.encode(cb, bb, true);
        return s.substring(0, cb.position());
    }

    /**
     * Сбалансированы ли скобки какого-то вида в строке.
     * <p>
     * Функция поддерживает только один вид скобок, заданный потребителем в виде пары
     * (openingBracket, closingBracket).
     * <p>
     * Функция не подходит потребителям, у которых скобки не помещаются в один 16-байтный символ.
     *
     * @return true, если скобки сбалансированы; false, если нет
     */
    static boolean areBracketsBalanced(String s, char openingBracket, char closingBracket) {
        int openingBracketsOccurredAndNotClosedSoFar = 0;
        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);

            if (c == openingBracket) {
                openingBracketsOccurredAndNotClosedSoFar++;
            } else if (c == closingBracket) {
                if (openingBracketsOccurredAndNotClosedSoFar <= 0) {
                    return false;
                }
                openingBracketsOccurredAndNotClosedSoFar--;
            }
        }

        return openingBracketsOccurredAndNotClosedSoFar == 0;
    }

    /**
     * Сбалансированы ли в строке квадратные скобки.
     */
    public static boolean areSquareBracketsBalanced(String s) {
        return areBracketsBalanced(s, '[', ']');
    }

    /**
     * Вырезает все теги и оставляет только содержимое
     *
     * @param htmlText строка содержащая валидный html
     * @return текст без тегов
     */
    public static String htmlToPlainText(String htmlText) {
        return htmlText.replaceAll("(?s)<[^>]*>(\\s*<[^>]*>)*", " ");
    }

    public static String joinLongsToString(Collection<Long> collection) {
        return collection.stream().map(Object::toString).collect(Collectors.joining(","));
    }

    public static String joinToString(Collection<?> collection) {
        return collection.stream().map(Object::toString).collect(Collectors.joining(", "));
    }

    public static List<Long> parseLongsFromString(String s) {
        return Stream.of(s.split(","))
                .map(String::trim)
                .map(Long::valueOf)
                .collect(toList());
    }

    /**
     * Возвращает все символы, встречающиеся в текстах.
     *
     * @param texts тексты
     * @return все символы, встречающиеся в текстах.
     */
    static String getAllDistinctChars(String... texts) {
        return stream(texts)
                .flatMapToInt(String::codePoints)
                .distinct()
                .collect(StringBuilder::new, StringBuilder::appendCodePoint, StringBuilder::append)
                .toString();
    }

    public static List<String> getDistinctStrings(List<String> strings) {
        return strings.stream().distinct().collect(toList());
    }

    /**
     * Возвращает все индексы вхождения строки {@code s} в {@code text}
     */
    public static List<Integer> getIndexesOfAllOccurrences(String text, String s) {
        int index = text.indexOf(s);
        List<Integer> indexes = new ArrayList<>();

        while (index >= 0) {
            indexes.add(index);
            index = text.indexOf(s, index + 1);
        }

        return indexes;
    }

    public static Set<String> splitStringToSet(@Nullable String str, String separator) {
        return splitString(str, separator, toSet());
    }

    public static List<String> splitStringToList(@Nullable String str, String separator) {
        return splitString(str, separator, toList());
    }

    public static <R> R splitString(@Nullable String str, String separator, Collector<String, ?, R> collector) {
        if (str == null) {
            return Stream.<String>empty().collect(collector);
        }

        return Stream.of(str.split(separator))
                .map(String::trim)
                .filter(x -> !x.isBlank())
                .collect(collector);
    }

    /**
     * Генерирует список уникальных имен для копирования, по правилам метода {@link #generateCopyName(String, String)}.
     *
     * @param originalNames список уже имеющихся имен
     * @param newNames список новых имен, которые, возможно, нужно модифицировать.
     * @param copyPrefix префикс копии, может быть null или пустым
     * @return список новых уникальных имен для копирования. Если изначальный список и так содержал уникальные имена,
     * то вернется он сам без модификаций.
     */
    public static List<String> generateCopyNames(
            Collection<String> originalNames, List<String> newNames, String copyPrefix) {
        if (newNames == null || newNames.isEmpty()) {
            return Collections.emptyList();
        }
        if (copyPrefix == null) {
            copyPrefix = "";
        }
        Set<String> originalNamesSet = new HashSet<>(originalNames);
        BitSet uniqueNamesFlags = new BitSet(newNames.size());
        // Сначала пройдемся по всем новым именам и проставим флаги для уникальных. Эти имена изменяться не будут.
        for (int i = 0; i < newNames.size(); i++) {
            String newName = newNames.get(i);
            if (originalNamesSet.add(newName)) {
                uniqueNamesFlags.set(i);
            }
        }
        // Если все имена уникальны, то просто возвращаем их как есть
        if (uniqueNamesFlags.cardinality() == newNames.size()) {
            return newNames;
        }
        // Сделаем карту для уникальных имен, где ключом будет имя без префикса копии и номера копии,
        // а значением - максимальный номер найденной копии
        Map<String, Integer> maxCopyNumbersByNames = new HashMap<>();
        for (String originalName: originalNamesSet) {
            addNameToCopyNumbersMap(maxCopyNumbersByNames, originalName, copyPrefix, false);
        }
        List<String> namesToReturn = new ArrayList<>(newNames.size());
        // Пройдемся по всем переданным новым именам, и добавим к ним префикс копии с номером копии, если нужно
        for (int i = 0; i < newNames.size(); i++) {
            String newName = newNames.get(i);
            if (uniqueNamesFlags.get(i)) {
                // Если имя было уникально - добавляем его как есть
                namesToReturn.add(newName);
            } else {
                // Если имя не было уникально, то добавляем его в карту с максимальным номером копии + 1,
                // и получаем координаты места вставки префикса и вычисленный номер копии
                long patternPositionAndCopyNumber =
                        addNameToCopyNumbersMap(maxCopyNumbersByNames, newName, copyPrefix, true);
                // Генерируем имя копии с префиксом и следующим номером копии (максимальный номер копии + 1)
                String generatedCopiedName = generateCopyName(newName, copyPrefix, patternPositionAndCopyNumber);
                namesToReturn.add(generatedCopiedName);
            }
        }
        return namesToReturn;
    }

    /**
     * Добавляет в карту максимальных номеров копий очередное имя, при этом увеличивая счетчик максимального
     * номера копии для "основы" имени.
     *
     * @param maxCopyNumbersByNames карта максимальных номеров копий по "основам" имен. Основа - это имя без
     *                              префикса копии и его номера.
     * @param originalName добавляемое в карту имя (возможно, с префиксом и каким-то номером копии)
     * @param copyPrefix префикс копии
     * @param increment нужно ли увеличить максимальный индекс на 1 при добавлении основы имени в карту.
     * @return Позиция вставки префикса и максимальный номер копирования до инкремента, запакованные в long.
     * Запаковка происходит аналогично таковой из {@link #getPatternPositionAndCopyNumberAsPacketLong(String, String)}
     * Передача номера копирования до инкремента удобна тем, что метод {@link #generateCopyName(String, String, long)}
     * автоматически увеличивает переданный номер копирования на 1, и передав ему результат этого метода мы сразу
     * получим правильное имя. Если передавать номер копирования после инкремента, то придется делать перепаковку,
     * что неудобно.
     */
    private static long addNameToCopyNumbersMap(
            Map<String, Integer> maxCopyNumbersByNames, String originalName, String copyPrefix, boolean increment) {
        int patternPosition = 0;
        int maxNumber;
        String key;
        if (originalName == null || originalName.isEmpty()) {
            key = "";
            maxNumber = maxCopyNumbersByNames.getOrDefault(key, 0);
        } else {
            long patternPositionAndCopyNumber =
                    getPatternPositionAndCopyNumberAsPacketLong(originalName, copyPrefix);
            int copyNumber = (int) patternPositionAndCopyNumber;
            if (copyNumber > 0) {
                patternPosition = (int) (patternPositionAndCopyNumber >> 32);
                key = originalName.substring(0, patternPosition);
            } else {
                patternPosition = originalName.length();
                key = originalName;
            }
            maxNumber = maxCopyNumbersByNames.getOrDefault(key, 0);
            maxNumber = Math.max(maxNumber, copyNumber);
        }
        maxCopyNumbersByNames.put(key, maxNumber + (increment ? 1 : 0));
        return (((long) patternPosition) << 32) | maxNumber;
    }

    /**
     * Генерирует имя для копирования. Для первичного копирования просто добавляет ' ($copyPrefix)' к originalName,
     * для повторных копирований изменяет в конце строки ' ($copyPrefix)' на ' ($copyPrefix $copyNumber)',
     * где $copyNumber растет с каждой копией. Например, вызов <code>generateCopyName("Баннер", "копия");</code> вернет
     * "Баннер (копия)", а вызов <code>generateCopyName("Баннер (копия)", "копия");</code> вернет
     * "Баннер (копия 2)". Вызов <code>generateCopyName("Баннер (копия 2)", "копия");</code> вернет
     * "Баннер (копия 3)", и так далее. Если номер копирования превысит {@link Integer#MAX_VALUE}, добавится новый
     * префикс и отсчет копий начнется снова.
     * @param originalName текущее имя, может быть null или пустым
     * @param copyPrefix префикс копии, может быть null или пустым
     * @return Имя для копирования
     */
    public static String generateCopyName(String originalName, String copyPrefix) {
        if (copyPrefix == null) {
            copyPrefix = "";
        }
        long patternPositionAndCopyNumber = getPatternPositionAndCopyNumberAsPacketLong(originalName, copyPrefix);
        return generateCopyName(originalName, copyPrefix, patternPositionAndCopyNumber);
    }

    /**
     * Генерирует имя для копирования, добавляя с позиции patternPosition префикс копирования copyPrefix и
     * номер копии сopyNumber.
     *
     * @param originalName оригинальное имя
     * @param copyPrefix префикс копии
     * @param patternPositionAndCopyNumber Позиция вставки префикса (patternPosition) и номер копирования (сopyNumber)
     *                                     запакованные в long. Старшие 32 бита - позиция, с которой нужно добавлять
     *                                     префикс, младшие 32 бита - номер копирования. Если это копирование - первое,
     *                                     то позиция должна быть равна 0, и номер копирования тоже быть равен 0
     * @return сгенерированное имя для копирования
     */
    @NotNull
    private static String generateCopyName(String originalName, String copyPrefix, long patternPositionAndCopyNumber) {
        int copyNumber = (int) patternPositionAndCopyNumber;
        if (copyNumber > 0) {
            int patternPosition = (int) (patternPositionAndCopyNumber >> 32);
            String namePrefix = patternPosition == 0 ? "" : originalName.substring(0, patternPosition) + " ";
            return copyPrefix.isEmpty() ?
                    namePrefix + '(' + (copyNumber + 1) + ')' :
                    namePrefix + '(' + copyPrefix + ' ' + (copyNumber + 1) + ')';
        } else {
            String namePrefix = originalName == null || originalName.isEmpty() ? "" : originalName + ' ';
            return copyPrefix.isEmpty() ?
                    namePrefix + "(1)" :
                    namePrefix + '(' + copyPrefix + ')';
        }
    }

    /**
     * Возвращает запакованные в long позицию вставки префикса и номер копирования $copyNumber из паттерна вида
     * (без кавычек) '($copyPrefix)' или '($copyPrefix $copyNumber)' или просто '($copyNumber)',
     * если copyPrefix null или пустая строка. Паттерн, если он есть, должен находиться в конце строки.
     * @param originalName строка для копирования
     * @param copyPrefix префикс копирования
     * @return Позиция вставки префикса и номер копирования запакованные в long. Старшие 32 бита позиция, с которой
     * нужно добавлять префикс, младшие 32 бита - номер копирования. Если это копирование - первое, то позиция будет
     * равна 0, и номер копирования тоже будет равен 0
     */
    private static long getPatternPositionAndCopyNumberAsPacketLong(String originalName, String copyPrefix) {
        if (originalName == null || originalName.isEmpty()) {
            return 0;
        }
        int length = originalName.length();
        if (originalName.charAt(length - 1) != ')') {
            return 0;
        }
        if (!copyPrefix.isEmpty()) {
            // Если префикс заканчивается на цифру, и если сначала пытаться искать цифру с конца,
            // то можем найти цифру из префикса, а не номер копирования, поэтому сначала проверим полный префикс,
            // и только если он не совпадет, пойдем искать номер копирования.
            int prefixBegin = length - copyPrefix.length() - 2;
            if (prefixBegin >= 0 && originalName.charAt(prefixBegin) == '(' &&
                    originalName.regionMatches(prefixBegin + 1, copyPrefix, 0, copyPrefix.length())) {
                prefixBegin = correctPatternProsition(prefixBegin, originalName);
                return (((long) prefixBegin) << 32) | 1;
            }
        }
        // Если префикс не совпал, возможно, после него идет число. Найдем его.
        int index = length - 2;
        long multiplier = 1;
        long copyNumber = 0;
        while (index >= 0) {
            char c = originalName.charAt(index);
            int digit = Character.digit(c, 10);
            if (digit < 0) {
                break;
            }
            copyNumber += digit * multiplier;
            if (copyNumber >= Integer.MAX_VALUE) {
                return 0;
            }
            multiplier *= 10L;
            index--;
        }
        boolean copyNumberNotExists = index == length - 2;
        // Если числа не было, и префикс не совпал, значит это точно не повторная копия.
        if (copyNumberNotExists) {
            return 0;
        }
        // Если префикс не пустой, и число нашлось, то проверяем что между ним и числом есть пробел
        if (!copyPrefix.isEmpty()) {
            if (originalName.charAt(index) != ' ') {
                return 0;
            }
            // откатываем индекс на начало префикса
            index -= copyPrefix.length() + 1;
        }
        // Проверяем, совпадает ли префикс, если он есть, и корректен ли найденный номер копии
        if (index >= 0 && originalName.charAt(index) == '(' && copyNumber > 0 &&
                (copyPrefix.isEmpty() ||
                        originalName.regionMatches(index + 1, copyPrefix, 0, copyPrefix.length()))) {
            // Если это не первая копия, и основа имени не пуста, то перед префиксом должен идти пробел. Учтем это.
            index = correctPatternProsition(index, originalName);
            return (((long) index) << 32) | copyNumber;
        } else {
            return 0;
        }
    }

    /**
     * Корректирует позицию, с которой надо начинать дописывать паттерн копии (префикс + номер копии).
     * Если перед prefixBegin стоит пробел, то нужно вернуться еще на один символ назад, так как это пробел,
     * разделяющий основу имени и префикс копии. Если основа имени пуста - то пробела перед префиксом копии не будет.
     *
     * @param prefixBegin координата символа, на котором найден префикс копирования
     * @param originalName оригинальное имя, возможно, с префиксом копирования и номером копии
     * @return скорректирванная позиция, с которой нужно начинать дописывать новый паттерн копии
     */
    private static int correctPatternProsition(int prefixBegin, String originalName) {
        if (prefixBegin > 0 && originalName.charAt(prefixBegin - 1) == ' ') {
            prefixBegin--;
        }
        return prefixBegin;
    }

}
