package ru.yandex.direct.utils;

import java.util.regex.Pattern;

import javax.annotation.Nullable;

import ru.yandex.direct.utils.hostname.CachedHostnameResolver;
import ru.yandex.direct.utils.hostname.FallbackHostnameResolver;
import ru.yandex.direct.utils.hostname.HostnameResolver;
import ru.yandex.direct.utils.hostname.SimpleHostnameResolver;

public class SystemUtils {
    private static final HostnameResolver STRICT_HOSTNAME_RESOLVER = new CachedHostnameResolver(
            new SimpleHostnameResolver()
    );

    private static final HostnameResolver FALLBACK_HOSTNAME_RESOLVER = new FallbackHostnameResolver(
            STRICT_HOSTNAME_RESOLVER,
            "localhost"
    );

    private static final Pattern YP_HOST_DC_PATTERN = Pattern.compile("^.*\\.([a-z]{3})\\d*\\.yp-c\\.yandex\\.net$");

    private SystemUtils() {
    }

    /**
     * Returns cached fully qualified host name or "localhost" if there was an error determining it.
     */
    public static String hostname() {
        return FALLBACK_HOSTNAME_RESOLVER.getHostname();
    }

    /**
     * Try to parse datacenter from hostname
     *
     * @return datacenter(f.e. sas) or null
     */
    @Nullable
    public static String hostDatacenter() {
        return parseDatacenter(hostname());
    }

    @Nullable
    public static String parseDatacenter(String hostname) {
        var matcher = YP_HOST_DC_PATTERN.matcher(hostname);
        if (matcher.matches()) {
            return matcher.group(1);
        } else {
            return null;
        }
    }

    /**
     * Returns cached fully qualified host name.
     */
    public static String strictHostname() {
        return STRICT_HOSTNAME_RESOLVER.getHostname();
    }

    static String cloudHostnameShortener(String host) {
        return host.replace(".vm.search.yandex.net", "")
                .replace(".yp-c.yandex.net", "")
                .replace(".gencfg-c.yandex.net", "");
    }

    public static String shortCloudHostname() {
        return cloudHostnameShortener(strictHostname());
    }

    /**
     * Returns process id
     */
    public static long getPid() {
        return ProcessHandle.current().pid();
    }
}
