package ru.yandex.direct.utils;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nullable;

import org.slf4j.Logger;

/**
 * Утилиты работы с потоками, частично повторена функциональность ThreadUtils из iceberg-misc
 */
public class ThreadUtils {
    private ThreadUtils() {
    }

    /**
     * Thread.sleep c отловом исключения InterruptedException и обернутым в InterruptedRuntimeException
     *
     * @param mills см {@link Thread#sleep(long)}
     */
    public static void sleep(long mills) {
        try {
            Thread.sleep(mills);
        } catch (InterruptedException e) {
            // Restore the interrupted status
            Thread.currentThread().interrupt();

            throw new InterruptedRuntimeException(e);
        }
    }

    /**
     * Вызывает {@link ThreadUtils##sleep(long)}
     *
     * @param duration длительность
     */
    public static void sleep(Duration duration) {
        sleep(duration.toMillis());
    }

    /**
     * Вызывает {@link ThreadUtils##sleep(long)}
     *
     * @param amount     длительность в chronoUnit единицах
     * @param chronoUnit единицы длительности sleep
     */
    public static void sleep(long amount, ChronoUnit chronoUnit) {
        sleep(Duration.of(amount, chronoUnit).toMillis());
    }

    public static void sleepUninterruptibly(Duration sleepDuration) {
        Interrupts.waitUninterruptibly(sleepDuration, Interrupts.waitMillisNanos(Thread::sleep));
    }

    /**
     * Ждет прерывания треда
     */
    @SuppressWarnings({"squid:S2189", "InfiniteLoopStatement"})
    public static void sleepUntilInterrupt() throws InterruptedException {
        while (true) {
            Thread.sleep(Long.MAX_VALUE);
        }
    }

    /**
     * Восстанавливает interrupted статус, если было выброшено InterruptedException
     * @return true, если передано InterruptedException
     */
    public static boolean checkInterrupted(Throwable e) {
        if (e instanceof InterruptedException) {
            Thread.currentThread().interrupt();
            return true;
        }
        return false;
    }

    /**
     * Запускает действие action с ретраями.
     *
     * @param maxAttempts  Максимальное кол-во попыток
     * @param startDelayMs Начальная задержка между попытками
     * @param mul          Коэффициент, на который домножается startDelayMs при каждой очередной неуспешной попытке,
     *                     >= 1.1
     * @param logger       Логгер (может быть null)
     */
    public static void execWithRetries(Consumer<Integer> action, int maxAttempts, long startDelayMs, double mul,
                                       @Nullable Logger logger) {
        execFuncWithRetries(attempt -> {
            action.accept(attempt);
            return null;
        }, maxAttempts, startDelayMs, mul, logger);
    }

    /**
     * Выполняет функцию func с ретраями.
     *
     * @param maxAttempts  Максимальное кол-во попыток
     * @param startDelayMs Начальная задержка между попытками
     * @param mul          Коэффициент, на который домножается startDelayMs при каждой очередной неуспешной попытке,
     *                     >= 1.1
     * @param logger       Логгер (может быть null)
     */
    public static <T> T execFuncWithRetries(Function<Integer, T> func, int maxAttempts, long startDelayMs, double mul,
                                            @Nullable Logger logger) {
        if (!(maxAttempts > 1)) {
            throw new IllegalArgumentException("maxAttempts should be > 1");
        }
        if (!(startDelayMs >= 1)) {
            throw new IllegalArgumentException("startDelayMs should be >= 1");
        }
        if (!(mul >= 1.1)) {
            throw new IllegalArgumentException("mul should be >= 1.1");
        }
        int attempt = 1;
        long delayMs = startDelayMs;
        while (attempt <= maxAttempts) {
            try {
                return func.apply(attempt);
            } catch (RuntimeException e) {
                if (attempt < maxAttempts) {
                    if (logger != null) {
                        logger.error(
                                String.format("Exception while retrying, attempt %d, will be retried", attempt), e);
                    }
                    attempt++;
                    try {
                        Thread.sleep(delayMs);
                    } catch (InterruptedException ex) {
                        Thread.currentThread().interrupt();
                        throw new InterruptedRuntimeException(e);
                    }
                    delayMs = (long) (delayMs * mul);
                } else {
                    if (logger != null) {
                        logger.error("Last retry failed", e);
                    }
                    throw e;
                }
            }
        }
        throw new AssertionError("Should not be here");
    }
}
