package ru.yandex.direct.utils;

/**
 * Этот класс помогает конструировать сложные AutoCloseable-объекты.
 * <p>
 * К примеру, хотим сделать класс, для работы с чистым временным инстансом mysql-сервера.
 * <p>
 * class TmpMysqlServer implements AutoCloseable {
 * ...
 * }
 * <p>
 * В этом классе нам нужно контроллировать два ресурса: процесс mysqld и временную директорию с данными.
 * <p>
 * Конструктор мог бы выглядеть так:
 * <p>
 * TmpMysqlServer() {
 * this.dir = createTmpDir();
 * initMysqlDataDir(this.dir);
 * this.server = startMysqld(this.dir);
 * }
 * <p>
 * В этом конструкторе есть проблема, если initMysqlDataDir или startMysqld завершатся с исключением, то
 * временная директория не будет удалена, поэтому правильней сделать так:
 * <p>
 * TmpMysqlServer() {
 * this.dir = createTmpDir();
 * try {
 * initMysqlDataDir(this.dir);
 * this.server = startMysqld(this.dir);
 * } catch (Throwable exc) {
 * dir.close();
 * throw exc;
 * }
 * }
 * <p>
 * Но с этим кодом тоже есть проблемы: dir.close() может бросить исключение, которое скроет оригинальное, поэтому
 * делаем так:
 * <p>
 * TmpMysqlServer() {
 * this.dir = createTmpDir();
 * try {
 * initMysqlDataDir(this.dir);
 * this.server = startMysqld(this.dir);
 * } catch (Throwable exc) {
 * try {
 * dir.close();
 * } catch (Throwable closeExc) {
 * exc.addSuppressed(closeExc);
 * }
 * throw exc;
 * }
 * }
 * <p>
 * Код уже слишком громоздкий, и уже есть неплохие шансы забыть написать throw exc.
 * А если кроме этого сразу после старта сервера нам нужно еще и выполнить на нем какой-нибудь запрос,
 * например, настроить репликацию, то таких вложенных try-catch блоков будет уже не один, а два.
 * <p>
 * С использованием Transient конструктор будет выглядеть так:
 * <p>
 * TmpMysqlServer() {
 * try (Transient\<TmpDir\> t = new Transient\<\>()) {
 * this.dir = t.item = createTmpDir();
 * initMysqlDataDir(this.dir);
 * this.server = startMysqld(this.dir);
 * t.success()
 * }
 * }
 */
public class Transient<T extends AutoCloseable> implements AutoCloseable {
    @SuppressWarnings("checkstyle:visibilitymodifier")
    public T item;

    public Transient() {
        this.item = null;
    }

    public Transient(T item) {
        this.item = item;
    }

    public void success() {
        if (item == null) {
            throw new IllegalStateException("Transient item is absent");
        }
        item = null;
    }

    /**
     * Вызывать только в качестве самой последней операции в try.
     * <p>
     * try (Transient<\?> t = ... ) {
     * ...
     * t.pop(); // <-- норм
     * }
     * <p>
     * try (Transient<\?> t = ... ) {
     * ...
     * list.add(t.pop()); // <-- не норм
     * }
     * <p>
     * Если не подходит, нужно использовать success().
     */
    public T pop() {
        if (item == null) {
            throw new IllegalStateException("Transient item is absent");
        }
        T result = item;
        item = null;
        return result;
    }

    @Override
    public void close() {
        if (item != null) {
            try {
                item.close();
            } catch (RuntimeException | Error exc) {
                throw exc;
            } catch (Exception exc) {
                throw new Checked.CheckedException(exc);
            } finally {
                item = null;
            }
        }
    }
}
