package ru.yandex.direct.utils;

import java.net.IDN;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.net.InternetDomainName;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.utils.model.UrlParts;

@ParametersAreNonnullByDefault
public class UrlUtils {
    private static final Pattern PORT_REGEX = Pattern.compile(":[0-9]+$");
    private static final Pattern PROTOCOL_PATTERN = Pattern.compile("(^https?)://");

    private UrlUtils() {
    }

    public static UrlParts laxParseUrl(String url) {
        return UrlParts.fromUrl(url);
    }

    /**
     * Удалить протокол http/https
     */
    public static String trimProtocol(String url) {
        return PROTOCOL_PATTERN.matcher(url).replaceFirst("");
    }

    /**
     * Получить протокол из url
     */
    public static String getProtocol(String url, String defaultProtocol) {
        var protocolMatcher = PROTOCOL_PATTERN.matcher(url);
        return protocolMatcher.find() ? protocolMatcher.group(1) : defaultProtocol;
    }

    public static String trimPort(String host) {
        return PORT_REGEX.matcher(host).replaceFirst("");
    }

    public static String urlDomainToAscii(String url) {
        UrlParts urlParts = laxParseUrl(url);
        return urlParts.toBuilder()
                .withDomain(IDN.toASCII(urlParts.getDomain()))
                .build()
                .toUrl();
    }

    public static String urlPathToAsciiIfCan(String urlStr) {
        try {
            String encodedUrl = encodeUrlIfCan(urlStr);
            UrlParts encodedUrlParts = laxParseUrl(encodedUrl);

            UrlParts urlParts = laxParseUrl(urlStr);
            return urlParts.toBuilder()
                    .withPath(encodedUrlParts.getPath())
                    .build()
                    .toUrl();
        } catch (Exception e) {
            return urlStr;
        }
    }

    public static String encodeUrlIfCan(String srcUrl) {
        try {
            URL url = new URL(srcUrl);
            URI uri = new URI(url.getProtocol(), url.getUserInfo(), url.getHost(), url.getPort(), url.getPath(),
                    url.getQuery(), url.getRef());
            return uri.toASCIIString();
        } catch (MalformedURLException | URISyntaxException e) {
            return srcUrl;
        }
    }

    public static String encodeUrlQueryIfCan(String srcUrl) {
        try {
            URL url = new URL(srcUrl);
            String query = url.getQuery();
            StringBuilder encodedQuery = new StringBuilder();

            if (query != null) {
                String[] params = query.split("&");
                for (int i = 0; i < params.length; i++) {
                    String[] param = params[i].split("=", 2);
                    encodedQuery.append(param[0]).append("=");
                    if (param.length == 2) {
                        encodedQuery.append(URLEncoder.encode(param[1], StandardCharsets.UTF_8));
                    }
                    if (i + 1 != params.length) {
                        encodedQuery.append("&");
                    }
                }
            }

            return getUrlFromParts(url.getProtocol(), url.getUserInfo(), url.getHost(), url.getPort(),
                    url.getPath(), encodedQuery.toString(), url.getRef());
        } catch (MalformedURLException e) {
            return srcUrl;
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    public static boolean isUrlInternal(String srcUrl) {
        String host;
        try {
            host = new URL(srcUrl).getHost();
            //Если проверяем кривой хост или IP, то мы все равно не сможем сказать, внутренний он или нет
            if (InternetDomainName.isValid(host)) {
                var topDomain = InternetDomainName.from(host).topPrivateDomain().toString();
                return topDomain.equals("yandex.net") || topDomain.equals("yandex-team.ru");
            }
        } catch (Exception ignored) {
        }
        return false;
    }

    public static String getUrlFromParts(String protocol, @Nullable String userInfo, String host,
                                         @Nullable Integer port, @Nullable String path, @Nullable String query,
                                         @Nullable String ref) {
        StringBuilder sb = new StringBuilder(protocol)
                .append("://");
        if (!StringUtils.isEmpty(userInfo)) {
            sb.append(userInfo)
                    .append("@");
        }
        sb.append(host);
        if (port != null && port > 0) {
            sb.append(":")
                    .append(port);
        }
        if (!StringUtils.isEmpty(path)) {
            sb.append(path);
        }
        if (!StringUtils.isEmpty(query)) {
            sb.append("?")
                    .append(query);
        }
        if (!StringUtils.isEmpty(ref)) {
            sb.append("#")
                    .append(ref);
        }

        return sb.toString();
    }
}
