package ru.yandex.direct.utils;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;

public class YamlUtils {
    private static final String DESERIALIZATION_ERROR = "can not deserialize object from YAML";

    public static final ObjectMapper MAPPER = new ObjectMapper(new YAMLFactory())
            .registerModule(new KotlinModule())
            .registerModule(new JavaTimeModule());

    public static final ObjectMapper MAPPER_WITHOUT_SPLIT_LINES =
            new ObjectMapper(new YAMLFactory().disable(YAMLGenerator.Feature.SPLIT_LINES))
                    .registerModule(new KotlinModule())
                    .registerModule(new JavaTimeModule());

    private YamlUtils() {
        // No instantiating, please
    }

    public static ObjectMapper getObjectMapper() {
        return MAPPER;
    }

    public static TypeFactory getTypeFactory() {
        return MAPPER.getTypeFactory();
    }

    /**
     * Десериализует объект из YAML
     *
     * @param yaml строка YAML
     * @param type тип объекта
     * @param <T>  тип объекта
     * @return десериализованный объект
     */
    public static <T> T fromYaml(String yaml, Class<T> type) {
        try {
            return MAPPER.readValue(yaml, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromYaml(String yaml, JavaType type) {
        try {
            return MAPPER.readValue(yaml, type);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> T fromYaml(String yaml, TypeReference<T> typeReference) {
        try {
            return MAPPER.readValue(yaml, typeReference);
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }
    }

    public static <T> String toYaml(T v) {
        try {
            return MAPPER.writeValueAsString(v);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to yaml", e);
        }
    }

    /*
    По дефолту длинные строки делятся на несколько частей при сериализации, что иногда мешает
     */
    public static <T> String toYamlWithoutSplitLines(T v) {
        try {
            return MAPPER_WITHOUT_SPLIT_LINES.writeValueAsString(v);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to yaml", e);
        }
    }
}
