package ru.yandex.direct.utils.collectors;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collector;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Объединяет набор Map, которые в своём значении содержат Set<V>. Находим все возможные ключи и для каждого
 * из них сливаем значения, которые хранятся по этому ключу, в один Set<V>
 */
@ParametersAreNonnullByDefault
public final class MergeMapCollector<K, V> implements Collector<Map<K, Set<V>>, Map<K, Set<V>>, Map<K, Set<V>>> {
    @Override
    public Supplier<Map<K, Set<V>>> supplier() {
        return HashMap::new;
    }

    private Map<K, Set<V>> mergeTwoMaps(Map<K, Set<V>> map1, Map<K, Set<V>> map2) {
        final var allKeys = new HashSet<>(map1.keySet());
        allKeys.addAll(map2.keySet());

        for (var key : allKeys) {
            final var accSet = map1.getOrDefault(key, new HashSet<>());
            final var nextSet = map2.get(key);

            if (nextSet != null) {
                accSet.addAll(nextSet);
            }

            map1.put(key, accSet);
        }

        return map1;
    }

    @Override
    public BiConsumer<Map<K, Set<V>>, Map<K, Set<V>>> accumulator() {
        return this::mergeTwoMaps;
    }

    @Override
    public BinaryOperator<Map<K, Set<V>>> combiner() {
        return this::mergeTwoMaps;
    }

    @Override
    public Function<Map<K, Set<V>>, Map<K, Set<V>>> finisher() {
        return acc -> acc;
    }

    @Override
    public Set<Characteristics> characteristics() {
        return Set.of(Characteristics.UNORDERED);
    }
}
