package ru.yandex.direct.utils.concurrency;

import java.time.Duration;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

/**
 * {@link ParallelBlockingProcessor}, который перебросит ошибку из упавшей задачи настолько рано, насколько возможно.
 * Ошибка бросается в следующих случаях:
 * <ul>
 * <li>При вызове {@link #spawn(Runnable)}, если точно известно, что хотя бы одна задача уже провалилась.</li>
 * <li>При вызове {@link #trySpawn(Runnable, Duration)}, если вызывающий поток сумел взять семафор и если точно
 * известно, что хотя бы одна задача уже провалилась.</li>
 * <li>При вызове {@link #close()}, при условии, что {@link #spawn(Runnable)} и {@link #trySpawn(Runnable, Duration)}
 * ни разу не бросали ошибку и при условии, что не был вызван {@link #setErrorWasChecked()}.</li>
 * </ul>
 */
@ParametersAreNonnullByDefault
@ThreadSafe
public class FailFastParallelBlockingProcessor extends ParallelBlockingProcessor {
    private final AtomicReference<RuntimeException> anyFail;
    private volatile boolean errorWasChecked;

    /**
     * @param parallel   Максимальное количество одновременно выполняющихся задач
     * @param namePrefix Префикс для имён потоков
     */
    public FailFastParallelBlockingProcessor(int parallel, String namePrefix) {
        super(parallel, namePrefix);
        anyFail = new AtomicReference<>();
    }

    @Override
    protected void beforeExecute() {
        checkAnyFail();
    }

    @Override
    protected void onError(RuntimeException e) {
        anyFail.compareAndSet(null, e);
    }

    private void checkAnyFail() {
        RuntimeException anyFailContent = anyFail.get();
        if (anyFailContent != null) {
            errorWasChecked = true;
            throw anyFailContent;
        }
    }

    /**
     * Препятствует дальнейшему бросанию ошибки в close(). Следует вызвать, если есть подозрения, что ошибка была, но
     * хочется её проигнорировать.
     */
    public void setErrorWasChecked() {
        errorWasChecked = true;
    }

    @Override
    public void close() {
        RuntimeException finalException = null;
        try {
            super.close();
        } catch (RuntimeException e) {
            finalException = e;
        }
        RuntimeException anyFailContent = anyFail.get();
        if (anyFailContent != null && !errorWasChecked) {
            if (finalException != null) {
                finalException.addSuppressed(anyFailContent);
            } else {
                finalException = anyFailContent;
            }
        }
        if (finalException != null) {
            throw finalException;
        }
    }
}
