package ru.yandex.direct.utils.concurrency;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import ru.yandex.direct.utils.NamedThreadFactory;

/**
 * Аналог ThreadPoolExecutor, изначально предназначенный для записи больших пачек данных. В отличие от
 * ThreadPoolExecutor, если ни один поток пока не готов немедленно приступить к выполнению задачи, то запрашивающий
 * поток будет заблокирован, пока не освободится хотя бы один рабочий поток. Это сделано в целях экономии памяти,
 * чтобы очередь задач не забивалась большими объектами.
 */
@ParametersAreNonnullByDefault
@ThreadSafe
public class ParallelBlockingProcessor implements AutoCloseable {
    private final Semaphore semaphore;
    private final ExecutorService pool;

    /**
     * @param parallel   Максимальное количество одновременно выполняющихся задач
     * @param namePrefix Префикс для имён потоков
     */
    public ParallelBlockingProcessor(int parallel, String namePrefix) {
        this.semaphore = new Semaphore(parallel);
        this.pool = Executors.newFixedThreadPool(parallel, new NamedThreadFactory(namePrefix));
    }

    /**
     * Запросить выполнение задачи в пуле потоков. Если все потоки заняты, то вызывающий поток будет заблокирован.
     *
     * @param command Задача, которую надо выполнить
     */
    public CompletableFuture<Void> spawn(Runnable command) throws InterruptedException {
        semaphore.acquire();
        return executeAndRelease(command);
    }

    /**
     * Запросить выполнение задачи в пуле потоков. Если все потоки заняты, то вызывающий поток будет заблокирован.
     *
     * @param command      Задача, которую надо выполнить
     * @param startTimeout Максимальное время ожидания начала выполнения задачи
     * @return CompletableFuture, если хотя бы один поток принялся выполнять задачу
     * @throws RuntimeException Если никто не начал в течение startTimeout
     */
    public CompletableFuture<Void> trySpawn(Runnable command, Duration startTimeout) throws InterruptedException {
        boolean acquired = semaphore.tryAcquire(startTimeout.toNanos(), TimeUnit.NANOSECONDS);
        if (acquired) {
            return executeAndRelease(command);
        } else {
            throw new RuntimeException(
                    "Waiting for ParallelBlockingProcessor thread too much. Timed out " + startTimeout);
        }
    }

    private CompletableFuture<Void> executeAndRelease(Runnable command) {
        try {
            beforeExecute();
            CompletableFuture<Void> future = new CompletableFuture<>();
            pool.execute(() -> {
                try {
                    if (!future.isCancelled()) {
                        command.run();
                        future.complete(null);
                    }
                } catch (RuntimeException e) {
                    future.completeExceptionally(e);
                    onError(e);
                } finally {
                    semaphore.release();
                }
            });
            return future;
        } catch (RuntimeException e) {
            // ошибка execute, задача даже не начинала выполняться
            semaphore.release();
            throw e;
        }
    }

    protected void onError(RuntimeException e) {
        // no hook by default
    }

    protected void beforeExecute() {
        // no hook by default
    }

    @Override
    public void close() {
        pool.shutdown();
    }
}
