package ru.yandex.direct.utils.converter;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collector;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;

/**
 * Все конверторы здесь null safe, т.е. если к ним на вход приходит значение null, оно не посылается на конвертацию,
 * а возвращается как есть.
 */
@ParametersAreNonnullByDefault
public class Converters {
    private Converters() {
    }

    public static <T> Converter<T, T> identity() {
        return v -> v;
    }

    public static <T1, T2> Converter<T1, T2> nullSafeConverter(Converter<T1, T2> converter) {
        return source -> source == null ? null : converter.convert(source);
    }

    /**
     * Конвертация по словарю значений
     */
    public static <S, D> Converter<S, D> mappingValueConverter(Map<S, D> valuesMap, @Nullable String errorOnAbsence) {
        String errorTemplate = (errorOnAbsence != null ? errorOnAbsence : "Can't convert value") + " %s";
        return nullSafeConverter(value -> {
            D convertedValue = valuesMap.get(value);
            if (convertedValue == null) {
                throw new IllegalStateException(String.format(errorTemplate, Objects.toString(value)));
            }
            return convertedValue;
        });
    }

    public static <S, D> Converter<S, D> mappingValueConverter(Map<S, D> valuesMap) {
        return mappingValueConverter(valuesMap, null);
    }

    public static <S, D> Converter<Collection<S>, Set<D>> immutableSetConverter(Map<S, D> valuesMap) {
        return immutableSetConverter(mappingValueConverter(valuesMap));
    }

    public static <S, D> Converter<Collection<S>, Set<D>> immutableSetConverter(Converter<S, D> valueConverter) {
        return setConverter(valueConverter, ImmutableSet.toImmutableSet());
    }

    private static <S, D> Converter<Collection<S>, Set<D>> setConverter(Converter<S, D> valueConverter,
                                                                        Collector<D, ?, ? extends Set<D>> collector) {
        return nullSafeConverter(valueSet -> valueSet.stream().map(valueConverter::convert).collect(collector));
    }
}
