package ru.yandex.direct.utils.crypt;

import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.lang3.RandomUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.copyOfRange;
import static org.apache.commons.lang3.ArrayUtils.addAll;
import static ru.yandex.direct.utils.HashingUtils.getMd5Hash;

/**
 * Реализация части методов из Perl'ового {@code Direct::Encrypt}
 * <p/>
 * Методы повторяют логику генерации {@code key} и {@code iv} (ключа и начального вектора) из {@code Crypt::CBC}.
 * Эта логика -- в {@link #genKeyAndIv(byte[], byte[])}.
 * Соль, как и {@code Crypt::CBC} храним в зашифрованном сообщении. Первые 16 байтов зашифрованного текста
 * должны содержать 8 байтов {@code "Salted__"} и 8 байтов самой соли.
 */
@SuppressWarnings("WeakerAccess")
public class Encrypter {
    /**
     * Префикс "солёного" шифра из perl {@code Crypt::CBC}
     */
    private static final byte[] SALTED_PREFIX = "Salted__".getBytes();
    /**
     * Длина используемой соли
     */
    private static final int SALT_LEN = 8;
    /**
     * Длина начального вектора. Дефолт из perl {@code Crypt::CBC}
     */
    private static final int IV_LEN = 16;
    /**
     * Длина ключа. Дефолт из perl {@code Crypt::CBC}
     */
    private static final int KEY_LEN = 32;
    /**
     * Используемый алгоритм
     */
    private static final String ALGORITHM = "AES/CBC/PKCS5Padding";

    private final byte[] pass;

    public Encrypter(String secret) {
        pass = secret.getBytes();
    }

    /**
     * Аналог Perl'ового
     * <pre>
     * sub decrypt_text
     * {
     *     my ($string) = @_;
     *     my $cipher = Crypt::CBC->new(
     *         -key => $secret,
     *         -cipher => 'Rijndael',
     *         -header => 'salt',
     *     );
     *     return $cipher->decrypt(pack "H*", $string);
     * }
     * </pre>
     * <p>
     * {@code Crypt::CBC} в сообщение добавляет байты с фразой {@code "Salted__"} и 8 байтами соли.
     * Эта реализация умеет из соли получать ключ и начальный вектор для декодирования
     *
     * @see #encryptText(String)
     */
    @SuppressWarnings("WeakerAccess")
    public String decryptText(String encrypted) {
        byte[] encodedBytes = pack(encrypted);
        // В шифре первые 8 байт -- слово "Salted__" (8 байт). После них 8 байт соль
        checkArgument(Arrays.equals(SALTED_PREFIX, copyOfRange(encodedBytes, 0, SALTED_PREFIX.length)),
                "Encoded text has unexpected prefix");
        int prefixAndSaltLen = SALTED_PREFIX.length + SALT_LEN;
        byte[] salt = copyOfRange(encodedBytes, SALTED_PREFIX.length, prefixAndSaltLen);
        // остальное -- шифрованное сообщение
        byte[] encryptedBytes = copyOfRange(encodedBytes, prefixAndSaltLen, encodedBytes.length);

        KeyAndIv keyAndIv = genKeyAndIv(pass, salt);

        byte[] resultBytes = decryptAes(encryptedBytes, keyAndIv.key, keyAndIv.iv);
        return new String(resultBytes);
    }

    /**
     * Расшифровка {@code encodedBytes} с использованием ключа {@code keyBytes} и начального вектора {@code ivBytes}
     */
    private static byte[] decryptAes(byte[] encryptedBytes, byte[] keyBytes, byte[] ivBytes) {
        try {
            SecretKeySpec secretKeySpec = new SecretKeySpec(keyBytes, "AES");
            Cipher cipher = Cipher.getInstance(ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, secretKeySpec, new IvParameterSpec(ivBytes));
            return cipher.doFinal(encryptedBytes);
        } catch (NoSuchAlgorithmException | NoSuchPaddingException | InvalidKeyException
                | BadPaddingException | IllegalBlockSizeException | InvalidAlgorithmParameterException e) {
            throw new EncryptionException("Error on decryption", e);
        }
    }

    /**
     * Аналог Perl'ового
     * <pre>
     * sub encrypt_text
     * {
     *     my ($string) = @_;
     *     my $cipher = Crypt::CBC->new(
     *         -key => $secret,
     *         -cipher => 'Rijndael',
     *         -header => 'salt',
     *     );
     *     return unpack "H*", $cipher->encrypt($string);
     * }
     * </pre>
     * <p>
     * По аналогии с {@code Crypt::CBC} добавляет в сообщение 16 байтов префикса:
     * фраза {@code "Salted__"} и 8 байтов соли.
     *
     * @see #decryptText(String)
     */
    public String encryptText(String text) {
        byte[] input = text.getBytes();
        byte[] salt = RandomUtils.nextBytes(SALT_LEN);

        byte[] result = encryptBytes(input, pass, salt);
        return unpack(result);
    }

    static byte[] encryptBytes(byte[] input, byte[] pass, byte[] salt) {
        KeyAndIv keyAndIv = genKeyAndIv(pass, salt);

        byte[] resultBytes = encryptAes(input, keyAndIv.key, keyAndIv.iv);
        return addAll(addAll(SALTED_PREFIX, salt), resultBytes);
    }

    private static byte[] encryptAes(byte[] encryptedBytes, byte[] keyBytes, byte[] ivBytes) {
        try {
            SecretKeySpec secretKeySpec = new SecretKeySpec(keyBytes, "AES");
            Cipher cipher = Cipher.getInstance(ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, secretKeySpec, new IvParameterSpec(ivBytes));
            return cipher.doFinal(encryptedBytes);
        } catch (NoSuchAlgorithmException | BadPaddingException | IllegalBlockSizeException
                | InvalidAlgorithmParameterException | InvalidKeyException | NoSuchPaddingException e) {
            // попасть сюда можем только если сломаем код
            throw new EncryptionException("Error on encryption", e);
        }
    }

    /**
     * Повторяет  perl'овый {@code unpack}.
     * <p/>
     * Возвращает строку с положительным hex-числом.
     * Примеры:
     * <pre>
     * A ([0x41]) -> 41
     * A! ([0x41, 0x21]) -> 4121
     * </pre>
     * Обратная операция: {@code new BigInteger(v, 16).toByteArray()}
     *
     * @see #pack(String)
     */
    static String unpack(byte[] a) {
        byte[] b;
        if ((a[0] & 0x80) != 0) {
            b = new byte[a.length + 1];
            System.arraycopy(a, 0, b, 1, a.length);
        } else {
            b = a;
        }
        BigInteger bigInteger = new BigInteger(b);
        return bigInteger.toString(16);
    }

    /**
     * Повторяет  perl'овый {@code pack}.
     * <p/>
     * Возвращает байты из строки с hex-числом.
     * Примеры:
     * <pre>
     * A -> [0x41]
     * A! -> [0x41, 0x21]
     * 0b11111111 -> [0xff]
     * </pre>
     *
     * @see #unpack(byte[])
     */
    static byte[] pack(String s) {
        byte[] bytes = new BigInteger(s, 16).toByteArray();
        if (bytes.length > s.length() / 2 && bytes[0] == 0 &&
                bytes.length > 1 && (bytes[1] & 0b10000000) != 0) {
            // старший бит был 1, поэтому в BigInteger добавлен лишний нулевой байт. Отрежем его
            return copyOfRange(bytes, 1, bytes.length);
        }
        return bytes;
    }

    /**
     * Повторена логика {@code Crypt::CBC::_salted_key_and_iv}
     */
    static KeyAndIv genKeyAndIv(byte[] pass, byte[] salt) {
        checkArgument(salt.length == SALT_LEN,
                String.format("Salt must be %d bytes long, actual: %d", SALT_LEN, salt.length));
        int desiredLen = KEY_LEN + IV_LEN;

        byte[] data = new byte[0];
        byte[] d = new byte[0];
        while (data.length < desiredLen) {
            byte[] valueToHash = addAll(d, addAll(pass, salt));
            d = getMd5Hash(valueToHash);
            data = addAll(data, d);
        }
        return new KeyAndIv(
                copyOfRange(data, 0, KEY_LEN),
                copyOfRange(data, KEY_LEN, desiredLen)
        );
    }

    static class KeyAndIv {
        final byte[] key;
        final byte[] iv;

        private KeyAndIv(byte[] key, byte[] iv) {
            this.key = key;
            this.iv = iv;
        }
    }
}
