package ru.yandex.direct.utils.io;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileUtils {
    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);

    private static final String USER_HOME_PREFIX = "~/";

    private FileUtils() {
    }

    /**
     * Атомарно пишет строку {@code contents} в файл {@code path}. Операция состоит из:
     * <ul>
     * <li>неатомарной записи во временный файл.</li>
     * <li>атомарного перемещения временного файла в {@code path}</li>
     * </ul>
     * Необходимо, чтобы родительский (по отношению к {@code path}) каталог был доступен для записи - для создания
     * временного файла. Необходимость создания временного файла в том же каталоге продиктована тем, что иначе
     * атомарность перемещения файла не гарантируется. (Точнее, она не гарантируется при перемещении файла
     * с одной файловой системы на другую. В пределах одной ФС перемещение атомарно)
     *
     * @param contents записываемая строка
     * @param path     записываемый файл
     * @throws RuntimeIoException
     */
    public static void atomicWrite(String contents, Path path) throws RuntimeIoException {
        try {
            File tmpFile = File.createTempFile(path.toString(), ".tmp", path.getParent().toFile());
            try {
                try (FileOutputStream fos = new FileOutputStream(tmpFile)) {
                    IOUtils.write(contents, fos, StandardCharsets.UTF_8);
                }
                Files.move(tmpFile.toPath(), path,
                        StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            } catch (IOException ex) {
                if (!tmpFile.delete()) {
                    logger.error("Can't delete temp file: {}", tmpFile);
                }
                throw ex;
            }
        } catch (IOException ex) {
            throw new RuntimeIoException(ex);
        }
    }

    /**
     * Прочитать текстовый файл в кодировку UTF-8 в строку
     *
     * @throws RuntimeIoException
     */
    public static String slurp(Path path) throws RuntimeIoException {
        try (InputStream inputStream = Files.newInputStream(path)) {
            return IOUtils.toString(inputStream, StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new RuntimeIoException(String.format("Can't slurp file %s", path), e);
        }
    }

    /**
     * Заменить {@code ~} в начале пути на домашний каталог.
     */
    public static Path expandHome(String path) {
        if (StringUtils.contains(path, USER_HOME_PREFIX)) {
            path = StringUtils.replaceOnce(path, USER_HOME_PREFIX, System.getProperty("user.home") + "/");
        }
        return Paths.get(path);
    }

    /**
     * Прочитать токен из файла
     */
    public static String readToken(Path path) {
        return FileUtils.slurp(path).trim();
    }
}
