package ru.yandex.direct.utils.io;

import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.Checked;

public class TempDirectory implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(TempDirectory.class);

    private final Path path;
    private boolean keep;

    public TempDirectory(String prefix) {
        try {
            path = Files.createTempDirectory(prefix + "-");
        } catch (IOException exc) {
            throw new Checked.CheckedException(exc);
        }
        keep = false;
    }

    public TempDirectory(Path dir, String prefix) {
        try {
            path = Files.createTempDirectory(dir, prefix + "-");
        } catch (IOException exc) {
            throw new Checked.CheckedException(exc);
        }
        keep = false;
    }

    /**
     * Recursive deletes the specified path
     */
    public static void deleteRecursive(Path path) {
        try {
            Files.walkFileTree(path, new SimpleFileVisitor<Path>() {
                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    Files.delete(dir);
                    return FileVisitResult.CONTINUE;
                }
            });
        } catch (IOException exc) {
            throw new Checked.CheckedException(exc);
        }
    }

    public Path getPath() {
        return path;
    }

    public void keep() {
        keep = true;
    }

    @Override
    public void close() {
        if (!keep) {
            logger.info("deleteRecursive for {}", path);
            deleteRecursive(path);
        }
    }

    @Override
    public String toString() {
        return "TempDirectory{" +
                "path=" + path +
                ", keep=" + keep +
                '}';
    }
}
