package ru.yandex.direct.utils.math;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import static com.google.common.base.Preconditions.checkState;

/**
 * Содержит различные математические утилиты
 */
public class MathUtils {
    private MathUtils() {
    }

    /**
     * Интерполирует координату Y точки, лежащей на ломаной, состоящей из нескольких отрезков, и
     * заданной набором точек
     *
     * @param pointX координата X интерполируемой точки
     * @param points набор точек, задающих ломаную. Точка должна определяться однозначно, то есть
     *               проекции отрезков на ось X не должны пересекаться между собой
     * @return точка, лежащая на ломаной, с заданной координатой X и интерполированной координатой Y
     */
    public static Point interpolateLinear(double pointX, List<Point> points) {
        if (points == null || points.isEmpty()) {
            throw new IllegalArgumentException("Cannot interpolate empty values");
        }
        if (points.size() == 1) {
            return points.get(0);
        }

        List<Point> sorted = new ArrayList<>(points);
        sorted.sort(Comparator.comparingDouble(Point::getX));
        Point left = sorted.get(0);

        for (int i = 1; i < sorted.size() - 1; i++) {
            Point right = sorted.get(i);

            if (pointX <= right.getX()) {
                return interpolateLinear(pointX, left, right);
            }

            left = right;
        }

        return interpolateLinear(pointX, left, sorted.get(sorted.size() - 1));
    }

    /**
     * Интерполирует координату Y точки, лежащей на прямой, по двум точкам, характеризующим эту прямую,
     * и координате X искомой точки
     *
     * @param pointX     координата X искомой точки
     * @param pointLeft  левая характерная точка прямой
     * @param pointRight правая характерная точка прямой
     * @return точка, лежащая на прямой, с указанной координатой X и интерполированной координатой Y
     */
    public static Point interpolateLinear(double pointX, Point pointLeft, Point pointRight) {
        double dy = pointRight.getY() - pointLeft.getY();
        double dx = pointRight.getX() - pointLeft.getX();
        double resultPointY;
        // если интерполируемая точка за пределами отрезка интерполяции, значение Y должно соответствовать
        // значению в ближайшем конце отрезка
        if (pointX < pointLeft.getX()) {
            resultPointY = pointLeft.getY();
        } else if (pointRight.getX() < pointX) {
            resultPointY = pointRight.getY();
        } else {
            resultPointY = pointLeft.getY() + (dy / dx * (pointX - pointLeft.getX()));
        }
        return Point.fromDoubles(pointX, resultPointY);
    }

    @SuppressWarnings("checkstyle:linelength")
    /**
     * Вычисление квантиля для списка значений по формуле
     * <a href="https://en.wikipedia.org/wiki/Percentile#Second_variant.2C_.7F.27.22.60UNIQ--postMath-00000043-QINU.60.22.27.7F">wikipedia</a><br/>
     * Повторяет логику perl Tools::calc_percentile
     *
     * @param quantile квантиль, от 0 до 1
     * @param <T>      тип значений в списке
     */
    public static <T extends Number> double calcQuantile(List<T> numberList, double quantile) {
        if (numberList.isEmpty()) {
            return 0;
        }
        checkState(quantile >= 0 && quantile <= 1, "quantile must be in range [0,1].");

        double[] sortedValues = numberList.stream()
                .mapToDouble(Number::doubleValue)
                .sorted()
                .toArray();

        int lastIndex = numberList.size() - 1;
        double percentileIndex = (double) lastIndex * quantile;
        int closestIntIndex = (int) percentileIndex;

        if (closestIntIndex == percentileIndex) {
            return sortedValues[closestIntIndex];
        }

        double fractionalPart = percentileIndex - (double) closestIntIndex;

        return sortedValues[closestIntIndex] +
                fractionalPart * (sortedValues[closestIntIndex + 1] - sortedValues[closestIntIndex]);
    }

    /**
     * Отдельная max-функция для улучшения читабельности формул
     */
    public static BigDecimal max(BigDecimal a, BigDecimal b) {
        return a.max(b);
    }

    /**
     * Отдельная min-функция для улучшения читабельности формул
     */
    public static BigDecimal min(BigDecimal a, BigDecimal b) {
        return a.min(b);
    }
}
