package ru.yandex.direct.utils.model;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

/**
 * части URL: URL можно разобрать на части и пересобрать из них
 * это лучше, чем {@link java.net.URL}, тем, что парсер менее строгий:
 * нужно для того, чтобы переваривать URL с шаблонами, которые задают клиенты
 * <p>
 * Пример: <code>http://example.com?var1={var1}&var2=var2</code> &mdash; неправильный
 * URL с точки зрения {@link java.net.URL}, но правильный с точки зрения
 * этого парсера
 */
@ParametersAreNonnullByDefault
public class UrlParts {
    // взято из RFC 3968
    private static final Pattern PATTERN =
            Pattern.compile("^(([^:/?#]+):)?(//([^/?#]*))?([^?#]*)(\\?([^#]*))?(#(.*))?");

    private final String protocol;
    private final String domain;
    private final String path;
    @Nullable
    private final List<Pair<String, String>> parameters;
    @Nullable
    private final String anchor;

    private UrlParts(String protocol, String domain, String path,
                     @Nullable List<Pair<String, String>> parameters, @Nullable String anchor) {
        this.protocol = protocol;
        this.domain = domain;
        this.path = path;
        this.parameters = parameters;
        this.anchor = anchor;
    }

    public static UrlParts fromUrl(String url) {
        Matcher matcher = PATTERN.matcher(url);
        if (matcher.matches()) {
            String protocol = matcher.group(2);
            String domain = matcher.group(4);
            String path = matcher.group(5);

            String encodedParameters = matcher.group(7);
            final List<Pair<String, String>> parameters = parseParameters(encodedParameters);

            String anchor = matcher.group(8);
            return new UrlParts(protocol, domain, path, parameters, anchor);
        } else {
            throw new IllegalArgumentException("invalid URL " + url);
        }
    }

    public String toUrl() {
        StringBuilder result = new StringBuilder(protocol + "://" + domain + path);

        if (parameters != null && !parameters.isEmpty()) {
            result.append("?");

            // URLEncodedUtils.format не подходит, потому что делает urlencode на драгоценные фигурные скобки
            result.append(parameters.stream()
                    .map(parameter -> parameter.getValue() != null ?
                            parameter.getKey() + "=" + parameter.getValue() :
                            parameter.getKey())
                    .collect(joining("&")));
        }

        if (anchor != null) {
            result.append(anchor);
        }

        return result.toString();
    }

    public String getProtocol() {
        return protocol;
    }

    public String getDomain() {
        return domain;
    }

    public String getPath() {
        return path;
    }

    @Nullable
    public List<Pair<String, String>> getParameters() {
        return parameters;
    }

    @Nullable
    public String getAnchor() {
        return anchor;
    }

    public Builder toBuilder() {
        return new Builder()
                .withProtocol(protocol)
                .withDomain(domain)
                .withPath(path)
                .withParameters(parameters)
                .withAnchor(anchor);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static List<Pair<String, String>> parseParameters(@Nullable String encodedParameters) {
        if (encodedParameters != null) {
            return Arrays.stream(encodedParameters.split("&"))
                    .map(nameAndValue -> {
                        String[] nameAndValueArray = nameAndValue.split("=", 2);
                        return nameAndValueArray.length == 2 ?
                                Pair.of(nameAndValueArray[0], nameAndValueArray[1]) :
                                Pair.of(nameAndValue, (String) null);
                    })
                    .filter(parameter -> StringUtils.isNotBlank(parameter.getKey()))
                    .collect(toList());
        }
        return null;
    }

    @Override
    public String toString() {
        return "UrlParts{" +
                "protocol='" + protocol + '\'' +
                ", domain='" + domain + '\'' +
                ", path='" + path + '\'' +
                ", definedParameters=" + parameters +
                ", anchor='" + anchor + '\'' +
                '}';
    }

    public static class Builder {
        private String protocol;
        private String domain;
        private String path;
        @Nullable
        private List<Pair<String, String>> parameters;
        @Nullable
        private String anchor;

        private Builder() {
        }

        public UrlParts build() {
            return new UrlParts(protocol, domain, path, parameters, anchor);
        }

        public Builder withProtocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        public Builder withDomain(String domain) {
            this.domain = domain;
            return this;
        }

        public Builder withPath(String path) {
            this.path = path;
            return this;
        }

        public Builder withParameters(@Nullable List<Pair<String, String>> parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder addParameters(String parametersToAdd) {
            var parsedParameters = parseParameters(parametersToAdd);
            return addParameters(parsedParameters);
        }

        public Builder addParameters(@Nullable List<Pair<String, String>> parameters) {
            if (this.parameters == null) {
                this.parameters = parameters;
            } else if (parameters != null) {
                this.parameters.addAll(parameters);
            }
            return this;
        }

        public Builder removeParamIf(String name, Predicate<String> predicate) {
            if (parameters == null) {
                return this;
            }

            List<Pair<String, String>> filtered = parameters.stream()
                    .filter(entry -> !(entry.getLeft().equalsIgnoreCase(name) && predicate.test(entry.getRight())))
                    .collect(toList());
            return withParameters(filtered);
        }

        private Builder addParam(String name, String value,
                                     Boolean addIfNotExists, Boolean addIfExists, Boolean removePrevious) {
            if (this.parameters == null) {
                if (addIfNotExists) {
                    return withParameters(List.of(Pair.of(name, value)));
                }
                return this;
            }
            List<Pair<String, String>> filtered = this.parameters.stream()
                    .filter(entry -> !entry.getLeft().equalsIgnoreCase(name))
                    .collect(toList());
            if (filtered.size() < parameters.size()) {
                if (!removePrevious) {
                    filtered = this.parameters;
                }
                if (addIfExists) {
                    filtered.add(Pair.of(name, value));
                }
                return withParameters(filtered);
            } else {
                if (addIfNotExists) {
                    filtered.add(Pair.of(name, value));
                    return withParameters(filtered);
                }
            }
            return this;
        }

        public Builder replaceParamIfExists(String name, String value) {
            return addParam(name, value, false, true, true);
        }

        public Builder replaceOrAddParam(String name, String value) {
            return addParam(name, value, true, true, true);
        }

        public Builder addParamIfNotExists(String name, String value) {
            return addParam(name, value, true, false, false);
        }

        public Builder addParameters(Map<String, String> parametersToAdd) {
            var parametersList = parametersToAdd.entrySet().stream()
                    .map(e -> Pair.of(e.getKey(), e.getValue()))
                    .collect(Collectors.toList());

            if (this.parameters == null) {
                this.parameters = parametersList;
            } else {
                this.parameters.addAll(parametersList);
            }
            return this;
        }

        public Builder withAnchor(@Nullable String anchor) {
            this.anchor = anchor;
            return this;
        }
    }
}
