package ru.yandex.direct.utils.net;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.escape.Escaper;
import com.google.common.net.UrlEscapers;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;

/**
 * Простой, но быстрый класс для формирования урлов.
 * Конструктор принимает префикс базового урла,
 * перегруженный addPath дописывает путь,
 * перегруженный addParam дописывает query-параметры.
 */
@ParametersAreNonnullByDefault
public class FastUrlBuilder {
    private static final Escaper PARAMETER_ESCAPER = UrlEscapers.urlFormParameterEscaper();

    private final StringBuilder sb;
    private boolean firstParam = true;

    public FastUrlBuilder(String url) {
        sb = new StringBuilder(url);
    }

    public FastUrlBuilder addPath(int path) {
        checkState(firstParam, "Can't append path, because first query param is already appended");
        sb.append('/').append(path);
        return this;
    }

    public FastUrlBuilder addPath(String path) {
        checkState(firstParam, "Can't append path, because first query param is already appended");
        sb.append('/').append(path);
        return this;
    }

    public FastUrlBuilder addParam(String name, String value) {
        checkNotNull(value);
        appendParamName(name);
        sb.append(PARAMETER_ESCAPER.escape(value));
        return this;
    }

    public FastUrlBuilder addParam(String name, int value) {
        appendParamName(name);
        sb.append(value);
        return this;
    }

    public FastUrlBuilder addParam(String name, long value) {
        appendParamName(name);
        sb.append(value);
        return this;
    }

    public FastUrlBuilder addParam(String name, char value) {
        appendParamName(name);
        sb.append(value);
        return this;
    }

    private void appendParamName(String name) {
        checkNotNull(name);
        char delimiter;
        if (firstParam) {
            firstParam = false;
            delimiter = '?';
        } else {
            delimiter = '&';
        }
        String escapedName = PARAMETER_ESCAPER.escape(name);
        sb.append(delimiter)
                .append(escapedName)
                .append('=');
    }

    public String build() {
        return sb.toString();
    }
}
