package ru.yandex.direct.utils.text;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.CharMatcher;
import org.apache.commons.lang3.StringUtils;

/**
 * Класс представляет удобный интерфейс для описания замен в тексте.
 * Рекомендуется к использованию вместо {@link String#replaceAll(String, String)}, если {@code pattern} не меняется.
 * В таком случае {@code replaceAll} будет напрасно на каждый вызов создавать экземлпяр {@link Pattern}.
 * <p>
 * Использование {@link StringModifier} позволяет описывать регулярные выражения, сохраняя связь с подстановками.
 * <p>
 * Можно использовать java.util регулярки или RE2 (для линейного времени)
 * </p>
 * <pre>
 * StringModifier stringModifier = new StringModifier.Builder()
 *     .withCustomRule(StringUtils::normalizeSpace)
 *     .withRegexpReplaceAllRule("[abc]+", "-")
 *     .build();
 * stringModifier.makeReplacements("10  cba 15 \t"); // return "10 - 15"
 * </pre>
 */
@ParametersAreNonnullByDefault
public class StringModifier {
    private final List<Function<String, String>> rules;

    private StringModifier(List<Function<String, String>> rules) {
        this.rules = new ArrayList<>(rules);
    }

    /**
     * Возвращает {@code text} с произведёнными заменами, которые хранятся в текущем {@link StringModifier}.
     */
    public String makeReplacements(@Nullable String text) {
        if (StringUtils.isEmpty(text)) {
            return text;
        }

        String result = text;
        for (Function<String, String> rule : rules) {
            result = rule.apply(result);
        }
        return result;
    }

    /**
     * Связывает {@link Pattern} с соответсвующей ему строковой заменой.
     */
    private static class SubstitutionRule implements Function<String, String> {
        private final Pattern pattern;
        private final String replacement;

        SubstitutionRule(Pattern pattern, String replacement) {
            this.pattern = pattern;
            this.replacement = replacement;
        }

        public String apply(String text) {
            return pattern.matcher(text).replaceAll(replacement);
        }
    }

    /**
     * Связывает RE2 {@link Pattern} с соответсвующей ему строковой заменой.
     */
    private static class Re2SubstitutionRule implements Function<String, String> {
        private final com.google.re2j.Pattern pattern;
        private final String replacement;

        Re2SubstitutionRule(com.google.re2j.Pattern pattern, String replacement) {
            this.pattern = pattern;
            this.replacement = replacement;
        }

        public String apply(String text) {
            return pattern.matcher(text).replaceAll(replacement);
        }
    }

    /**
     * Builder для {@link StringModifier}.
     * Порядок описания правил преобразования сохраняется при создании {@link StringModifier}.
     */
    public static class Builder {

        private final List<Function<String, String>> rules = new ArrayList<>();

        /**
         * Добавляет правило для замены всех подстрочек, найденных регулярным выражением {@code pattern},
         * на строку {@code replacement}.
         *
         * @see #withRegexpReplaceAllRule(String, String)
         */
        public Builder withRegexpReplaceAllRule(Pattern pattern, String replacement) {
            SubstitutionRule rule = new SubstitutionRule(pattern, replacement);
            rules.add(rule);
            return this;
        }

        /**
         * Добавляет правило по замену регулярного выражения на результат выполнения функции
         */
        public Builder withRegexpReplaceAllRule(Pattern pattern, Function<Matcher, String> replacer) {
            rules.add(s -> {
                Matcher m = pattern.matcher(s);
                StringBuilder sb = new StringBuilder();
                while (m.find()) {
                    m.appendReplacement(sb, replacer.apply(m));
                }
                m.appendTail(sb);
                return sb.toString();
            });
            return this;
        }

        /**
         * Добавляет правило по замену регулярного выражения на результат выполнения функции
         */
        public Builder withRegexpReplaceAllRule(String pattern, Function<Matcher, String> replacer) {
            return withRegexpReplaceAllRule(Pattern.compile(pattern), replacer);
        }

        /**
         * Добавляет правило для замены всех подстрочек, найденных регулярным выражением {@code pattern},
         * на строку {@code replacement}
         *
         * @see #withRegexpReplaceAllRule(Pattern, String)
         */
        public Builder withRegexpReplaceAllRule(String pattern, String replacement) {
            return withRegexpReplaceAllRule(Pattern.compile(pattern), replacement);
        }

        /**
         * Добавляет правило для замены всех подстрочек, найденных регулярным выражением {@code pattern},
         * на строку {@code replacement}.
         *
         * @see #withRegexpReplaceAllRule(String, String)
         */
        public Builder withRe2ReplaceAllRule(com.google.re2j.Pattern pattern, String replacement) {
            var rule = new Re2SubstitutionRule(pattern, replacement);
            rules.add(rule);
            return this;
        }

        /**
         * Добавляет правило по замену регулярного выражения на результат выполнения функции
         */
        public Builder withRe2ReplaceAllRule(com.google.re2j.Pattern pattern, Function<com.google.re2j.Matcher,
                String> replacer) {
            rules.add(s -> {
                var m = pattern.matcher(s);
                StringBuilder sb = new StringBuilder();
                while (m.find()) {
                    m.appendReplacement(sb, replacer.apply(m));
                }
                m.appendTail(sb);
                return sb.toString();
            });
            return this;
        }

        /**
         * Добавляет правило по замену регулярного выражения на результат выполнения функции
         */
        public Builder withRe2ReplaceAllRule(String pattern, Function<com.google.re2j.Matcher, String> replacer) {
            return withRe2ReplaceAllRule(com.google.re2j.Pattern.compile(pattern), replacer);
        }

        /**
         * Добавляет правило для замены всех подстрочек, найденных регулярным выражением {@code pattern},
         * на строку {@code replacement}
         *
         * @see #withRegexpReplaceAllRule(Pattern, String)
         */
        public Builder withRe2ReplaceAllRule(String pattern, String replacement) {
            return withRe2ReplaceAllRule(com.google.re2j.Pattern.compile(pattern), replacement);
        }

        /**
         * Добавляет произвольное правило преобразования строки
         */
        public Builder withCustomRule(Function<String, String> rule) {
            rules.add(rule);
            return this;
        }

        /**
         * Добавляет правило замены всех последовательностей из символов, входящих в chars на replacement
         */
        public Builder withCollapseAnyOf(CharSequence chars, char replacement) {
            return withCharMatcherCollapse(CharMatcher.anyOf(chars), replacement);
        }

        /**
         * Добавляет правило замены всех последовательностей, подходящих под CharMatcher на replacement
         *
         * @see CharMatcher::collapseFrom
         */
        public Builder withCharMatcherCollapse(CharMatcher matcher, char replacement) {
            rules.add(s -> matcher.collapseFrom(s, replacement));
            return this;
        }

        /**
         * Добавляет правило удаления всех символов, входящих в chars
         */
        public Builder withRemoveAnyOf(CharSequence chars) {
            return withCharMatcherRemove(CharMatcher.anyOf(chars));
        }

        /**
         * Добавляет правило удаления всех символов, подходящих под charMatcher
         */
        public Builder withCharMatcherRemove(CharMatcher charMatcher) {
            rules.add(charMatcher::removeFrom);
            return this;
        }

        /**
         * @return экземпляр {@link StringModifier}
         */
        public StringModifier build() {
            return new StringModifier(rules);
        }
    }
}
