package ru.yandex.direct.utils.text;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;

import static java.util.Map.entry;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.TextConstants.ENG_ALPHABET;
import static ru.yandex.direct.utils.TextConstants.LETTERS;


/**
 * Сервис для транслитерации строк
 */
public class TranslitService {

    private static final Set<Character> NON_LATIN_LETTERS = nonLatinLetters();

    private static final Map<Character, String> TRANSLIT_TABLE = translateTable();

    protected TranslitService() {
        // prevents calls from subclass
        throw new UnsupportedOperationException();
    }

    /**
     * Транслитерирует русские буквы входной строки
     * Удаляет все буквы не русского и латинского алфавитов
     * Аналог https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Translit.pm?rev=7177001#L42, только
     * без удаления пробелов
     */

    public static String translit(String input) {
        if (Objects.isNull(input)) {
            return null;
        }
        StringBuilder stringBuilder = new StringBuilder();
        for (char ch : input.toCharArray()) {
            if (TRANSLIT_TABLE.containsKey(ch)) {
                stringBuilder.append(TRANSLIT_TABLE.get(ch));
            } else if (!NON_LATIN_LETTERS.contains(ch)) {
                stringBuilder.append(ch);
            }
        }
        return stringBuilder.toString();

    }

    private static Set<Character> nonLatinLetters() {
        Set<Character> latinLetters = ENG_ALPHABET.chars()
                .mapToObj(code -> (char) code)
                .collect(toSet());
        Set<Character> allLetters = LETTERS.chars()
                .mapToObj(code -> (char) code)
                .collect(toSet());
        return Sets.difference(allLetters, latinLetters);
    }

    private static Map<Character, String> translateTable() {
        var lowerCaseLettersMap = Map.ofEntries(
                entry('а', "a"),
                entry('б', "b"),
                entry('в', "v"),
                entry('г', "g"),
                entry('д', "d"),
                entry('е', "e"),
                entry('ё', "yo"),
                entry('ж', "zh"),
                entry('з', "z"),
                entry('и', "i"),
                entry('й', "y"),
                entry('к', "k"),
                entry('л', "l"),
                entry('м', "m"),
                entry('н', "n"),
                entry('о', "o"),
                entry('п', "p"),
                entry('р', "r"),
                entry('с', "s"),
                entry('т', "t"),
                entry('у', "u"),
                entry('ф', "f"),
                entry('х', "kh"),
                entry('ц', "ts"),
                entry('ч', "ch"),
                entry('ш', "sh"),
                entry('щ', "shch"),
                entry('ы', "y"),
                entry('э', "e"),
                entry('ю', "yu"),
                entry('я', "ya"),
                entry('ъ', ""),
                entry('ь', ""));

        var upperCaseLettersMap = lowerCaseLettersMap.entrySet().stream()
                .collect(toMap(
                        e -> Character.toUpperCase(e.getKey()),
                        e -> StringUtils.capitalize(e.getValue())
                ));

        Map<Character, String> resultMap = new HashMap<>();
        resultMap.putAll(lowerCaseLettersMap);
        resultMap.putAll(upperCaseLettersMap);
        return Map.copyOf(resultMap);
    }
}
