package ru.yandex.direct.validation;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.LocalDateTime;
import java.time.YearMonth;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import com.google.common.base.Strings;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.validator.routines.EmailValidator;

public class Predicates {

    // common

    public static <T, C extends Collection<T>> Predicate<C> each(Predicate<T> predicate) {
        return value -> value.stream().allMatch(predicate);
    }

    public static <T> Predicate<T> not(Predicate<T> predicate) {
        return predicate.negate();
    }

    public static <T> Predicate<T> ignore() {
        return value -> true;
    }

    public static <T> Predicate<T> isNull() {
        return Objects::isNull;
    }

    public static <T> Predicate<T> notNull() {
        return Objects::nonNull;
    }

    public static <T> Predicate<T> inSet(Set<T> set) {
        return set::contains;
    }

    public static <T> Predicate<T> notInSet(Set<T> set) {
        return value -> !set.contains(value);
    }

    // numbers

    public static Predicate<Long> inRange(Long min, Long max) {
        return value -> value >= min && value <= max;
    }

    public static Predicate<Integer> inRange(Integer min, Integer max) {
        return value -> value >= min && value <= max;
    }

    public static Predicate<BigDecimal> inRange(BigDecimal min, BigDecimal max) {
        return value -> value.compareTo(min) >= 0 && value.compareTo(max) <= 0;
    }

    public static Predicate<Long> greaterThan(long border) {
        return value -> value > border;
    }

    public static Predicate<BigInteger> greaterThan(BigInteger border) {
        return value -> value.compareTo(border) > 0;
    }

    public static Predicate<Long> greaterThanOrEqualTo(long border) {
        return value -> value >= border;
    }

    public static Predicate<Integer> greaterThan(int border) {
        return value -> value > border;
    }

    public static Predicate<Long> lessThan(long border) {
        return value -> value < border;
    }

    public static Predicate<Integer> lessThan(int border) {
        return value -> value < border;
    }

    // strings

    public static Predicate<String> empty() {
        return String::isEmpty;
    }

    public static Predicate<String> notEmpty() {
        return str -> !str.isEmpty();
    }

    public static Predicate<String> blank() {
        return StringUtils::isBlank;
    }

    @SuppressWarnings("Convert2MethodRef")
    public static Predicate<String> nullOrEmpty() {
        return t -> Strings.isNullOrEmpty(t);
    }

    public static Predicate<String> length(int min, int max) {
        return value -> value.length() >= min && value.length() <= max;
    }

    public static Predicate<String> minLength(int min) {
        return value -> value.length() >= min;
    }

    public static Predicate<String> maxLength(int max) {
        return value -> value.length() <= max;
    }


    @SuppressWarnings("Duplicates")
    public static Predicate<String> isLong() {
        return s -> {
            try {
                //noinspection ResultOfMethodCallIgnored
                Long.valueOf(s);
                return true;
            } catch (NumberFormatException e) {
                return false;
            }
        };
    }

    @SuppressWarnings("Duplicates")
    public static Predicate<String> isInteger() {
        return s -> {
            try {
                //noinspection ResultOfMethodCallIgnored
                Integer.valueOf(s);
                return true;
            } catch (NumberFormatException e) {
                return false;
            }
        };
    }

    public static Predicate<String> isDouble() {
        return s -> {
            try {
                //noinspection ResultOfMethodCallIgnored
                Double.valueOf(s);
                return true;
            } catch (NumberFormatException e) {
                return false;
            }
        };
    }

    private static final Pattern IS_POSITIVE_WHOLE_NUMBER = Pattern.compile("[\\d]+");

    public static Predicate<String> isPositiveWholeNumber() {
        return s -> IS_POSITIVE_WHOLE_NUMBER.matcher(s).matches();
    }

    public static Predicate<String> validDateTime(DateTimeFormatter formatter) {
        return dateTimeStr -> {
            try {
                LocalDateTime.parse(dateTimeStr, formatter);
                return true;
            } catch (DateTimeParseException e) {
                return false;
            }
        };
    }

    public static Predicate<String> validYearMonth(DateTimeFormatter formatter) {
        return dateStr -> {
            try {
                YearMonth.parse(dateStr, formatter);
                return true;
            } catch (DateTimeParseException e) {
                return false;
            }
        };
    }

    public static Predicate<String> validEmail() {
        return str -> EmailValidator.getInstance().isValid(str);
    }

    public static Predicate<String> notContains(List<String> chars) {
        return s -> StreamEx.of(chars).noneMatch(s::contains);
    }

    // collections
    public static <I> Predicate<List<I>> listSize(int min, int max) {
        return list -> list.size() >= min && list.size() <= max;
    }

    public static <I> Predicate<Set<I>> setSize(int min, int max) {
        return set -> set.size() >= min && set.size() <= max;
    }

    public static <I extends Map> Predicate<I> mapSize(int min, int max) {
        return map -> map.size() >= min && map.size() <= max;
    }

    public static <I extends Enum<I>> Predicate<EnumSet<I>> enumSetSize(int min, int max) {
        return set -> set.size() >= min && set.size() <= max;
    }

    @SuppressWarnings("Convert2MethodRef")
    public static <I> Predicate<I> in(Set<I> set) {
        return value -> set.contains(value);
    }

    public static <I extends Collection> Predicate<I> notEmptyCollection() {
        return c -> !c.isEmpty();
    }
}
