package ru.yandex.direct.validation.builder;

import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

/**
 * Констрейнт: принимает на вход элемент, а возвращает либо null, если все ок,
 * либо дефект, если проверяемый элемент не соответствует ограничению.
 * <p>
 * Является точкой расширения: методы проверок в ItemValidationBuilder и ListValidationBuilder
 * принимают интерфейс {@code Constraint}, который можно имплементировать в клиентском коде.
 * Сама библиотека не содержит ни специфических дефектов, ни констрейнтов.
 * <p>
 * Для удобства создания констрейнта есть фабричный метод fromPredicate.
 *
 * @param <T> тип проверяемого элемента
 * @param <D> тип дефекта
 */
public interface Constraint<T, D> extends Function<T, D> {
    /**
     * Получает из предиката констрейнт, который возращает дефект,
     * если в параметр предиката передано значение не равное <code>null</code>
     * и не выполнился предикат <code>predicate</code>.
     * В противном случае, возращает <code>null</code>.
     *
     * @param predicate Условие выполнения
     * @param defect    Дефект, который порождается, если условие не выполняется
     * @param <T>       Тип параметра предиката
     * @param <D>       Тип дефекта
     * @return Сгенерированный констрейнт
     */
    static <T, D> Constraint<T, D> fromPredicate(Predicate<T> predicate, D defect) {
        return value -> (value == null || predicate.test(value)) ? null : defect;
    }

    static <T, D> Constraint<T, D> fromPredicate(Predicate<T> predicate, Function<T, D> defectCreator) {
        return value -> (value == null || predicate.test(value)) ? null : defectCreator.apply(value);
    }

    static <T, D> Constraint<T, D> fromPredicateAndDefectSupplier(Predicate<T> predicate, Supplier<D> supplier) {
        return value -> (value == null || predicate.test(value)) ? null : supplier.get();
    }

    /**
     * Получает из предиката констрейнт, который возвращает дефект, если не выполнился предикат. Не обрабатывает
     * дополнительно null, позволяя это сделать предикату. Полезно для тех случаев, когда null является значимым
     * случаем - проверки "не null", "не null и не пустая строка", и т.п.
     *
     * @param predicate Условие выполнения
     * @param defect    Дефект, который порождается, если условие не выполняется
     * @param <T>       Тип параметра предиката
     * @param <D>       Тип дефекта
     * @return Сгенерированный констрейнт
     */
    static <T, D> Constraint<T, D> fromPredicateOfNullable(Predicate<T> predicate, D defect) {
        return value -> predicate.test(value) ? null : defect;
    }

    /**
     * Сделать констрейнт на основе предиката этого констрейнта, но с другим дефектом.
     *
     * @param newDefect дефект нового констрейнта
     * @return Новый констрейнт с аналогичным предикатом, но с новым заданным дефектом.
     */
    default Constraint<T, D> overrideDefect(D newDefect) {
        return fromPredicate(value -> this.apply(value) == null, newDefect);
    }
}
