package ru.yandex.direct.validation.builder;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Класс, непосредственно используемый в клиентском коде для валидации объектов.
 * <p>
 * Пример использования.
 * <pre> {@code
 *
 * // валидируемый класс
 * public class Address {
 *     private String city;
 *     private Integer postcode;
 *
 *     // getters/setters
 * }
 *
 * // сервис валидации
 * public class AddressValidationService {
 *
 *     public ValidationResult<Address, DefectType> validate(Address address) {
 *         Set<String> allCities = Cities.getAvailableCities();
 *         Set<Integer> allPostcodes = Postcodes.getAvailablePostcodes();
 *
 *         ItemValidationBuilder<Address, DefectType> validationBuilder =
 *                 ItemValidationBuilder.of(address);
 *
 *         validationBuilder.check(cityAndPostcodeIsConsistent());
 *
 *         validationBuilder.item(address.getCity(), "city")
 *                 .check(notNull())
 *                 .check(notEmpty())
 *                 .check(inSet(allCities), When.isValid());
 *
 *         validationBuilder.item(address.getPostcode(), "postcode")
 *                 .check(notNull())
 *                 .check(greaterThan(0))
 *                 .check(inSet(allPostcodes), When.isValid());
 *
 *         return validationBuilder.getResult();
 *     }
 * }
 * }</pre>
 *
 * @param <T> тип валидируемого объекта
 * @param <D> тип дефекта
 */
@ParametersAreNonnullByDefault
public class ItemValidationBuilder<T, D> {

    private final ValidationResult<T, D> validationResult;

    public ItemValidationBuilder(ValidationResult<T, D> validationResult) {
        this.validationResult = validationResult;
    }

    public static <T, D> ItemValidationBuilder<T, D> of(@Nullable T item) {
        return new ItemValidationBuilder<>(new ValidationResult<>(item));
    }

    public static <T, D> ItemValidationBuilder<T, D> of(@Nullable T item,
                                                        @SuppressWarnings("unused") Class<D> defectType) {
        return of(item);
    }

    public ValidationResult<T, D> getResult() {
        return validationResult;
    }

    public <F> ItemValidationBuilder<F, D> item(@Nullable F value, String name) {
        return new ItemValidationBuilder<>(
                validationResult.getOrCreateSubValidationResult(PathHelper.field(name), value));
    }

    public <F> ListValidationBuilder<F, D> list(@Nullable List<F> value, String name) {
        return new ListValidationBuilder<>(
                validationResult.getOrCreateSubValidationResult(PathHelper.field(name), value));
    }

    public ItemValidationBuilder<T, D> check(Constraint<T, D> constraint) {
        return internalCheck(constraint, null, null, validationResult::addError);
    }

    public ItemValidationBuilder<T, D> check(Constraint<T, D> constraint, D overrideDefect) {
        return internalCheck(constraint, overrideDefect, null, validationResult::addError);
    }

    public ItemValidationBuilder<T, D> check(Constraint<T, D> constraint, D overrideDefect, When<T, D> when) {
        return internalCheck(constraint, overrideDefect, when, validationResult::addError);
    }

    public ItemValidationBuilder<T, D> check(Constraint<T, D> constraint, When<T, D> when) {
        return internalCheck(constraint, null, when, validationResult::addError);
    }

    public ItemValidationBuilder<T, D> weakCheck(Constraint<T, D> constraint) {
        return internalCheck(constraint, null, null, validationResult::addWarning);
    }

    public ItemValidationBuilder<T, D> weakCheck(Constraint<T, D> constraint, D overrideDefect) {
        return internalCheck(constraint, overrideDefect, null, validationResult::addWarning);
    }

    public ItemValidationBuilder<T, D> weakCheck(Constraint<T, D> constraint, D overrideDefect, When<T, D> when) {
        return internalCheck(constraint, overrideDefect, when, validationResult::addWarning);
    }

    public ItemValidationBuilder<T, D> weakCheck(Constraint<T, D> constraint, When<T, D> when) {
        return internalCheck(constraint, null, when, validationResult::addWarning);
    }

    public ItemValidationBuilder<T, D> checkBy(Validator<? extends T, D> validator) {
        return internalCheckBy(validator, null);
    }

    public ItemValidationBuilder<T, D> checkBy(Validator<? extends T, D> validator, When<T, D> when) {
        return internalCheckBy(validator, when);
    }

    public ItemValidationBuilder<T, D> checkByFunction(Function<T, D> validator) {
        Objects.requireNonNull(validator, "validator");
        return internalCheckByFunction(validator, null);
    }

    public ItemValidationBuilder<T, D> checkByFunction(Function<T, D> validator, When<T, D> when) {
        Objects.requireNonNull(validator, "validator");
        return internalCheckByFunction(validator, when);
    }

    protected ItemValidationBuilder<T, D> internalCheck(Constraint<T, D> constraint,
                                                        @Nullable D overrideDefect, @Nullable When<T, D> when, Consumer<D> defectConsumer) {
        T value = validationResult.getValue();
        if (when == null || when.apply(validationResult)) {
            BuilderUtils.check(value, constraint, overrideDefect, defectConsumer);
        }
        return this;
    }

    @SuppressWarnings("unchecked")
    protected <V extends T> ItemValidationBuilder<T, D> internalCheckBy(Validator<V, D> validator, @Nullable When<T, D> when) {
        T value = validationResult.getValue();
        if (when == null || when.apply(validationResult)) {
            ValidationResult<V, D> vr = validator.apply((V) value);
            this.validationResult.merge(vr);
        }
        return this;
    }

    protected ItemValidationBuilder<T, D> internalCheckByFunction(Function<T, D> validator, @Nullable When<T, D> when) {
        internalCheckBy(
                v -> Optional.ofNullable(validator.apply(v))
                        .map(d -> ValidationResult.failed(v, d))
                        .orElseGet(() -> ValidationResult.success(v)),
                when);
        return this;
    }
}
