package ru.yandex.direct.validation.builder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.BiPredicate;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Класс, непосредственно используемый в клиентском коде для валидации списков.
 * <p>
 * Пример использования.
 * <pre> {@code
 *
 * // тип элементов списка
 * public class Address {
 *     private String city;
 *     private Integer postcode;
 *
 *     // getters/setters
 * }
 *
 * // сервис валидации
 * public class AddressValidationService {
 *
 *     public ValidationResult<List<Address>, DefectType> validate(List<Address> addressList) {
 *         Set<String> allCities = Cities.getAvailableCities();
 *         Set<Integer> allPostcodes = Postcodes.getAvailablePostcodes();
 *
 *         ListValidationBuilder<Address, DefectType> validationBuilder =
 *                 ListValidationBuilder.of(addressList);
 *
 *         validationBuilder
 *                 .check(notNull())
 *                 .check(minListSize(1))
 *                 .check(maxListSize(MAX_ADDRESS_LIST_SIZE))
 *                 .checkEachBy(notNull())
 *                 .checkEachBy(this::validateAddress, When.isValid());
 *
 *         return validationBuilder.getResult();
 *     }
 *
 *     public ValidationResult<Address, DefectType> validateAddress(Address address) {
 *         Set<String> allCities = Cities.getAvailableCities();
 *         Set<Integer> allPostcodes = Postcodes.getAvailablePostcodes();
 *
 *         ItemValidationBuilder<Address, DefectType> validationBuilder =
 *                 ItemValidationBuilder.of(address);
 *
 *         validationBuilder.check(cityAndPostcodeIsConsistent());
 *
 *         validationBuilder.item(address.getCity(), "city")
 *                 .check(notNull())
 *                 .check(notEmpty())
 *                 .check(inSet(allCities), When.isValid());
 *
 *         validationBuilder.item(address.getPostcode(), "postcode")
 *                 .check(notNull())
 *                 .check(greaterThan(0))
 *                 .check(inSet(allPostcodes), When.isValid());
 *
 *         return validationBuilder.getResult();
 *     }
 * }
 * }</pre>
 *
 * @param <T> тип валидируемого объекта
 * @param <D> тип дефекта
 */
@ParametersAreNonnullByDefault
public class ListValidationBuilder<T, D> {
    private ValidationResult<List<T>, D> validationResult;

    public ListValidationBuilder(ValidationResult<List<T>, D> validationResult) {
        this.validationResult = validationResult;
    }

    public static <T, D> ListValidationBuilder<T, D> of(@Nullable List<T> list) {
        return new ListValidationBuilder<>(new ValidationResult<>(list));
    }

    public static <T, D> ListValidationBuilder<T, D> of(@Nullable List<T> list,
                                                        @SuppressWarnings("unused") Class<D> defectType) {
        return of(list);
    }

    public ValidationResult<List<T>, D> getResult() {
        return validationResult;
    }

    public ListValidationBuilder<T, D> check(Constraint<List<T>, D> constraint) {
        return internalCheck(constraint, null, null, validationResult::addError);
    }

    public ListValidationBuilder<T, D> check(Constraint<List<T>, D> constraint, D overrideDefect) {
        return internalCheck(constraint, overrideDefect, null, validationResult::addError);
    }

    public ListValidationBuilder<T, D> check(Constraint<List<T>, D> constraint, D overrideDefect,
                                             When<List<T>, D> when) {
        return internalCheck(constraint, overrideDefect, when, validationResult::addError);
    }

    public ListValidationBuilder<T, D> check(Constraint<List<T>, D> constraint, When<List<T>, D> when) {
        return internalCheck(constraint, null, when, validationResult::addError);
    }

    public ListValidationBuilder<T, D> weakCheck(Constraint<List<T>, D> constraint) {
        return internalCheck(constraint, null, null, validationResult::addWarning);
    }

    public ListValidationBuilder<T, D> weakCheck(Constraint<List<T>, D> constraint, D overrideDefect) {
        return internalCheck(constraint, overrideDefect, null, validationResult::addWarning);
    }

    public ListValidationBuilder<T, D> weakCheck(Constraint<List<T>, D> constraint, D overrideDefect,
                                                 When<List<T>, D> when) {
        return internalCheck(constraint, overrideDefect, when, validationResult::addWarning);
    }

    public ListValidationBuilder<T, D> weakCheck(Constraint<List<T>, D> constraint, When<List<T>, D> when) {
        return internalCheck(constraint, null, when, validationResult::addWarning);
    }

    public ListValidationBuilder<T, D> checkBy(Validator<List<T>, D> validator) {
        return internalCheckBy(validator, null);
    }

    public ListValidationBuilder<T, D> checkBy(Validator<List<T>, D> validator, When<List<T>, D> when) {
        return internalCheckBy(validator, when);
    }

    public ListValidationBuilder<T, D> checkEach(Constraint<T, D> constraint) {
        return internalCheckEach(constraint, null, null, ValidationResult::addError);
    }

    public ListValidationBuilder<T, D> checkEach(Constraint<T, D> constraint, D overrideDefect) {
        internalCheckEach(constraint, overrideDefect, null, ValidationResult::addError);
        return this;
    }

    public ListValidationBuilder<T, D> checkEach(Constraint<T, D> constraint, D overrideDefect, When<T, D> when) {
        internalCheckEach(constraint, overrideDefect, when, ValidationResult::addError);
        return this;
    }

    public ListValidationBuilder<T, D> checkEach(Constraint<T, D> constraint, When<T, D> when) {
        internalCheckEach(constraint, null, when, ValidationResult::addError);
        return this;
    }

    public ListValidationBuilder<T, D> checkEach(ListConstraint<T, D> constraint) {
        internalCheckEach(constraint, null, ValidationResult::addError, null);
        return this;
    }

    public ListValidationBuilder<T, D> checkEach(ListConstraint<T, D> constraint, D overrideDefect) {
        internalCheckEach(constraint, overrideDefect, ValidationResult::addError, null);
        return this;
    }

    public ListValidationBuilder<T, D> checkEach(ListConstraint<T, D> constraint, When<T, D> when) {
        internalCheckEach(constraint, null, ValidationResult::addError, when);
        return this;
    }

    public ListValidationBuilder<T, D> checkEach(ListConstraint<T, D> constraint, D overrideDefect, When<T, D> when) {
        internalCheckEach(constraint, overrideDefect, ValidationResult::addError, when);
        return this;
    }

    public ListValidationBuilder<T, D> weakCheckEach(Constraint<T, D> constraint) {
        return internalCheckEach(constraint, null, null, ValidationResult::addWarning);
    }

    public ListValidationBuilder<T, D> weakCheckEach(Constraint<T, D> constraint, D overrideDefect) {
        internalCheckEach(constraint, overrideDefect, null, ValidationResult::addWarning);
        return this;
    }

    public ListValidationBuilder<T, D> weakCheckEach(Constraint<T, D> constraint, D overrideDefect, When<T, D> when) {
        internalCheckEach(constraint, overrideDefect, when, ValidationResult::addWarning);
        return this;
    }

    public ListValidationBuilder<T, D> weakCheckEach(Constraint<T, D> constraint, When<T, D> when) {
        internalCheckEach(constraint, null, when, ValidationResult::addWarning);
        return this;
    }

    public ListValidationBuilder<T, D> weakCheckEach(ListConstraint<T, D> constraint) {
        internalCheckEach(constraint, null, ValidationResult::addWarning, null);
        return this;
    }

    public ListValidationBuilder<T, D> weakCheckEach(ListConstraint<T, D> constraint, When<T, D> when) {
        internalCheckEach(constraint, null, ValidationResult::addWarning, when);
        return this;
    }

    public ListValidationBuilder<T, D> weakCheckEach(ListConstraint<T, D> constraint, D overrideDefect) {
        internalCheckEach(constraint, overrideDefect, ValidationResult::addWarning, null);
        return this;
    }

    public ListValidationBuilder<T, D> weakCheckEach(ListConstraint<T, D> constraint, D overrideDefect,
                                                     When<T, D> when) {
        internalCheckEach(constraint, overrideDefect, ValidationResult::addWarning, when);
        return this;
    }


    public ListValidationBuilder<T, D> checkEachBy(Validator<T, D> validator) {
        return internalCheckEachBy(validator, null);
    }

    public ListValidationBuilder<T, D> checkEachBy(Validator<T, D> validator, When<T, D> when) {
        return internalCheckEachBy(validator, when);
    }

    public ListValidationBuilder<T, D> checkEachBy(ListItemValidator<T, D> validator) {
        return internalCheckEach(validator, null);
    }

    public ListValidationBuilder<T, D> checkEachBy(ListItemValidator<T, D> validator, When<T, D> when) {
        return internalCheckEach(validator, when);
    }

    /**
     * Отфильтровать sublist по условию when, передать валидатору списка,
     * запомнив соответствие индексов исходного и переданного списков.
     * Получив результат валидации смерджить его по соответствующим нодам.
     *
     * @param validator функция, возвращающая результат валидации по подсписку и мапе его текущих индексов в старые
     * @param when      условие фильтрации списка для передачи в валидатор
     * @return Этот же инстанс {@link ListValidationBuilder}.
     */
    public ListValidationBuilder<T, D> checkSublistBy(
            BiFunction<List<T>, Map<Integer, Integer>, ValidationResult<List<T>, D>> validator,
            When<T, D> when) {
        checkNotNull(when, "\"when\" is required");
        return checkSublistBy(validator, (vr, ind) -> when.apply(vr));
    }

    /**
     * Отфильтровать sublist по условию when, передать валидатору списка,
     * запомнив соответствие индексов исходного и переданного списков.
     * Получив результат валидации смерджить его по соответствующим нодам.
     *
     * @param validator      функция, возвращающая результат валидации по подсписку и мапе его текущих индексов в старые
     * @param checkPredicate условие фильтрации списка по значению и индексу элементадля передачи в валидатор
     * @return Этот же инстанс {@link ListValidationBuilder}.
     */
    public ListValidationBuilder<T, D> checkSublistBy(
            BiFunction<List<T>, Map<Integer, Integer>, ValidationResult<List<T>, D>> validator,
            BiPredicate<ValidationResult<T, D>, Integer> checkPredicate) {
        checkNotNull(checkPredicate, "\"checkPredicate\" is required");
        List<T> list = getValue();
        if (list == null) {
            return this;
        }
        List<T> validatedSublist = new ArrayList<>();
        Map<Integer, Integer> indexMap = new HashMap<>();
        for (int i = 0; i < list.size(); i++) {
            T item = list.get(i);
            ValidationResult<T, D> itemVr = validationResult.getOrCreateSubValidationResult(index(i), item);
            if (checkPredicate.test(itemVr, i)) {
                validatedSublist.add(item);
                indexMap.put(validatedSublist.size() - 1, i);
            }
        }
        if (validatedSublist.isEmpty()) {
            return this;
        }
        ValidationResult<List<T>, D> sublistVr = validator.apply(validatedSublist, indexMap);
        sublistVr.getErrors().forEach(validationResult::addError);
        sublistVr.getWarnings().forEach(validationResult::addWarning);
        for (int j = 0; j < validatedSublist.size(); j++) {
            T item = validatedSublist.get(j);
            ValidationResult<T, D> sublistItemVr = sublistVr.getOrCreateSubValidationResult(index(j), item);
            int sourceIndex = indexMap.get(j);
            ValidationResult<T, D> itemVr = validationResult.getOrCreateSubValidationResult(index(sourceIndex), item);
            itemVr.merge(sublistItemVr);
        }
        return this;

    }

    /**
     * Отфильтровать sublist по условию when, передать валидатору списка,
     * запомнив соответствие индексов исходного и переданного списков.
     * Получив результат валидации смерджить его по соответствующим нодам.
     *
     * @param validator валидатор списка, которому будует передан список объектов,
     *                  удовлетворяющих условию when
     * @param when      условие фильтрации списка для передачи в валидатор
     * @return Этот же инстанс {@link ListValidationBuilder}.
     */
    public ListValidationBuilder<T, D> checkSublistBy(Validator<List<T>, D> validator, When<T, D> when) {
        return checkSublistBy((l, m) -> validator.apply(l), when);
    }

    /**
     * Отфильтровать sublist по условию на элемент и индекс элемента, передать валидатору списка,
     * запомнив соответствие индексов исходного и переданного списков.
     * Получив результат валидации смерджить его по соответствующим нодам.
     *
     * @param validator      валидатор списка, которому будует передан список объектов,
     *                       удовлетворяющих условию when
     * @param checkPredicate условие фильтрации списка для передачи в валидатор
     * @return Этот же инстанс {@link ListValidationBuilder}.
     */
    public ListValidationBuilder<T, D> checkSublistBy(Validator<List<T>, D> validator,
                                                      BiPredicate<ValidationResult<T, D>, Integer> checkPredicate) {
        return checkSublistBy((l, m) -> validator.apply(l), checkPredicate);
    }

    protected List<T> getValue() {
        return validationResult.getValue();
    }

    protected ListValidationBuilder<T, D> internalCheck(Constraint<List<T>, D> constraint,
                                                        @Nullable D overrideDefect, @Nullable When<List<T>, D> when,
                                                        Consumer<D> defectConsumer) {
        if (when == null || when.apply(validationResult)) {
            BuilderUtils.check(getValue(), constraint, overrideDefect, defectConsumer);
        }
        return this;
    }

    protected ListValidationBuilder<T, D> internalCheckBy(Validator<List<T>, D> validator,
                                                          @Nullable When<List<T>, D> when) {
        if (when == null || when.apply(validationResult)) {
            ValidationResult<List<T>, D> vr = validator.apply(getValue());
            this.validationResult.merge(vr);
        }
        return this;
    }

    protected ListValidationBuilder<T, D> internalCheckEach(Constraint<T, D> constraint, @Nullable D overrideDefect,
                                                            @Nullable When<T, D> when, BiConsumer<ValidationResult<T,
            D>, D> defectConsumer) {
        List<T> list = getValue();
        if (list == null) {
            return this;
        }
        for (int i = 0; i < list.size(); i++) {
            T item = list.get(i);
            ValidationResult<T, D> itemVr = validationResult.getOrCreateSubValidationResult(index(i), list.get(i));
            if (when == null || when.apply(itemVr)) {
                BuilderUtils.check(item, constraint, overrideDefect, d -> defectConsumer.accept(itemVr, d));
            }
        }
        return this;
    }

    protected void internalCheckEach(ListConstraint<T, D> constraint, @Nullable D overrideDefect,
                                     BiConsumer<ValidationResult<T, D>, D> defectConsumer, @Nullable When<T, D> when) {
        List<T> list = getValue();
        if (list == null) {
            return;
        }
        List<T> validatedItems;
        Map<Integer, Integer> indexMap;
        if (when == null) {
            validatedItems = list;
            indexMap = null;
        } else {
            // если задано условие when фильтрующее элементы, нам надо запомнить их в отдельный список
            validatedItems = new ArrayList<>(list.size());
            // и запомнить маппинг индексов в старом и новом списках
            indexMap = new HashMap<>();
            for (int i = 0; i < list.size(); i++) {
                T item = list.get(i);
                ValidationResult<T, D> itemVr = validationResult.getOrCreateSubValidationResult(index(i), item);
                if (when.apply(itemVr)) {
                    validatedItems.add(item);
                    indexMap.put(validatedItems.size() - 1, i);
                }
            }
        }
        Map<Integer, D> defects = constraint.apply(validatedItems);
        // проходим по индексам списка, страхуясь от IndexOutOfBound в случае некорректно сформированного defects
        for (int i = 0; i < validatedItems.size(); i++) {
            D defect = defects.get(i);
            if (defect != null) {
                Integer originalIdx = indexMap == null ? i : indexMap.get(i);
                ValidationResult<T, D> itemVr =
                        validationResult.getOrCreateSubValidationResult(index(originalIdx), list.get(originalIdx));
                defectConsumer.accept(itemVr, overrideDefect != null ? overrideDefect : defect);
            }
        }
    }

    protected ListValidationBuilder<T, D> internalCheckEachBy(Validator<T, D> validator, @Nullable When<T, D> when) {
        List<T> list = getValue();
        if (list == null) {
            return this;
        }
        for (int i = 0; i < list.size(); i++) {
            T item = list.get(i);
            ValidationResult<T, D> itemVr = validationResult.getOrCreateSubValidationResult(index(i), item);
            if (when == null || when.apply(itemVr)) {
                ValidationResult<T, D> vr = validator.apply(item);
                itemVr.merge(vr);
            }
        }
        return this;
    }

    protected ListValidationBuilder<T, D> internalCheckEach(
            ListItemValidator<T, D> validator, @Nullable When<T, D> when) {
        List<T> list = getValue();
        if (list == null) {
            return this;
        }
        for (int i = 0; i < list.size(); i++) {
            T item = list.get(i);
            ValidationResult<T, D> itemVr = validationResult.getOrCreateSubValidationResult(index(i), item);
            if (when == null || when.apply(itemVr)) {
                ValidationResult<T, D> vr = validator.validate(i, item);
                itemVr.merge(vr);
            }
        }
        return this;
    }
}
