package ru.yandex.direct.validation.builder;

import java.util.function.Predicate;
import java.util.function.Supplier;

import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Принимает решение, стоит или нет выполнять некоторую проверку.
 * Принимает на вход результат валидации, тем самым получая  доступ
 * к валидируемому значению и найденным для него ошибкам и предупреждениям.
 * <p>
 *
 * @param <T> тип валидируемого значения
 * @param <D> тип дефекта
 */
public class When<T, D> {
    @SuppressWarnings("rawtypes")
    private static final When IS_VALID = new When<>(vr -> !vr.hasAnyErrors());

    @SuppressWarnings("rawtypes")
    private static final When NOT_NULL = new When<>(vr -> vr.getValue() != null);

    @SuppressWarnings("rawtypes")
    private static final When ALWAYS = new When<>(vr -> true);

    @SuppressWarnings("rawtypes")
    private static final When NEVER = new When<>(vr -> false);

    private final Predicate<ValidationResult<T, D>> predicate;

    When(Predicate<ValidationResult<T, D>> predicate) {
        this.predicate = predicate;
    }

    @SuppressWarnings("unchecked")
    public static <T, D> When<T, D> isValid() {
        return IS_VALID;
    }

    @SuppressWarnings("unchecked")
    public static <T, D> When<T, D> notNull() {
        return NOT_NULL;
    }

    public static <T, D> When<T, D> notNullAnd(When<T, D> when) {
        return new When<>(vr -> vr.getValue() != null && when.apply(vr));
    }

    public static <T, D> When<T, D> isValidAnd(When<T, D> when) {
        return new When<>(vr -> !vr.hasAnyErrors() && when.apply(vr));
    }

    public static <T, D> When<T, D> valueIs(Predicate<T> predicate) {
        return new When<>(vr -> predicate.test(vr.getValue()));
    }

    public static <T, D> When<T, D> isTrue(Supplier<Boolean> supplier) {
        return new When<>(vr -> supplier.get());
    }

    @SuppressWarnings("unchecked")
    public static <T, D> When<T, D> isTrue(boolean condition) {
        return condition ? ALWAYS : NEVER;

    }

    @SuppressWarnings("unchecked")
    public static <T, D> When<T, D> isFalse(boolean condition) {
        return condition ? NEVER : ALWAYS;
    }

    public static <T, D> When<T, D> isValidBoth(ItemValidationBuilder builder) {
        return new When<>(vr -> !(vr.hasAnyErrors() || builder.getResult().hasAnyErrors()));
    }

    //todo переименовать в check
    public boolean apply(ValidationResult<T, D> validationResult) {
        return predicate.test(validationResult);
    }
}
