package ru.yandex.direct.validation.constraint;

import java.util.Collection;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListConstraint;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.validation.Predicates.each;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class CollectionConstraints {

    private CollectionConstraints() {
    }


    // collections
    public static <C extends Collection> Constraint<C, Defect> notEmptyCollection() {
        return fromPredicate(Predicates.notEmptyCollection(), CollectionDefects.notEmptyCollection());
    }

    public static <C extends Collection> Constraint<C, Defect> isEmptyCollection() {
        return fromPredicate(Collection::isEmpty, CollectionDefects.isEmptyCollection());
    }

    public static <I> Constraint<List<I>, Defect> listSize(int min, int max) {
        return fromPredicate(Predicates.listSize(min, max), CollectionDefects.collectionSizeInInterval(min, max));
    }

    public static <I> Constraint<List<I>, Defect> collectionSize(int min, int max) {
        return fromPredicate(Predicates.listSize(min, max), CollectionDefects.collectionSizeIsValid(min, max));
    }

    public static <I> Constraint<List<I>, Defect> minListSize(int min) {
        return fromPredicate(Predicates.listSize(min, Integer.MAX_VALUE), CollectionDefects.minCollectionSize(min));
    }

    public static <I> Constraint<List<I>, Defect> maxListSize(int max) {
        return fromPredicate(Predicates.listSize(0, max), CollectionDefects.maxCollectionSize(max));
    }

    public static <I> Constraint<Set<I>, Defect> setSize(int min, int max) {
        return fromPredicate(Predicates.setSize(min, max), CollectionDefects.collectionSizeInInterval(min, max));
    }

    public static <I> Constraint<Set<I>, Defect> minSetSize(int min) {
        return fromPredicate(Predicates.setSize(min, Integer.MAX_VALUE), CollectionDefects.minCollectionSize(min));
    }

    public static <I> Constraint<Set<I>, Defect> maxSetSize(int max) {
        return fromPredicate(Predicates.setSize(0, max), CollectionDefects.maxCollectionSize(max));
    }

    public static <I extends Map> Constraint<I, Defect> maxMapSize(int max) {
        return fromPredicate(Predicates.mapSize(0, max), CollectionDefects.maxCollectionSize(max));
    }

    public static <I extends Map> Constraint<I, Defect> notEmptyMap() {
        return fromPredicate(not(Map::isEmpty), CollectionDefects.notEmptyCollection());
    }

    public static <I extends Enum<I>> Constraint<EnumSet<I>, Defect> maxEnumSetSize(int max) {
        return fromPredicate(Predicates.enumSetSize(0, max), CollectionDefects.maxCollectionSize(max));
    }

    public static <T> ListConstraint<T, Defect> unique() {
        return new SimpleUniqueItemsConstraint<>();
    }

    public static <T> ListConstraint<T, Defect> unique(Function<T, ?> getter) {
        return new SimpleUniqueItemsConstraint<>(getter);
    }

    /**
     * Позволяет найти дубликаты с помощью переданного {@code comparator}'а.
     * Дубликатами считаются элементы, для которых {@link Comparator#compare(Object, Object) comparator.compare(..)}
     * возвращает {@code 0}.
     *
     * @param comparator {@link Comparator}
     * @return экземпляр {@link ListConstraint} валидирующий список на наличие дубликатов
     */
    public static <T> ListConstraint<T, Defect> unique(Comparator<T> comparator) {
        return new UniqueComparableItemsConstraint<>(comparator);
    }

    /**
     * Каждое число в проверяемой коллекции -- корректный идентификатор (>0)
     */
    public static <C extends Collection<Long>> Constraint<C, Defect<?>> eachValidId() {
        return fromPredicate(each(not(Predicates.lessThan(1L))), CommonDefects.validId());
    }
}
