package ru.yandex.direct.validation.constraint;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

public class CommonConstraints {

    private CommonConstraints() {
    }

    public static <T> Constraint<T, Defect> success() {
        return v -> null;
    }

    public static <T, D> Constraint<T, Defect> unconditional(Defect<D> defect) {
        return Constraint.fromPredicateOfNullable(Objects::isNull, defect);
    }

    public static <T> Constraint<T, Defect> isNull() {
        return Constraint.fromPredicateOfNullable(Objects::isNull, CommonDefects.isNull());
    }

    public static <T> Constraint<T, Defect> notNull() {
        return Constraint.fromPredicateOfNullable(Objects::nonNull, CommonDefects.notNull());
    }

    public static <T> Constraint<T, Defect> isEqual(T b, Defect defect) {
        return fromPredicate(a -> Objects.equals(a, b), defect);
    }

    public static <T> Constraint<T, Defect> notEqual(T b, Defect defect) {
        return fromPredicate(a -> !Objects.equals(a, b), defect);
    }

    public static <T extends Collection<?>> Constraint<T, Defect> isEqualCollection(T c, Defect defect) {
        return fromPredicate(a -> CollectionUtils.isEqualCollection(a, c), defect);
    }

    public static Constraint<Boolean, Defect> isTrue() {
        return Constraint.fromPredicateOfNullable(Boolean.TRUE::equals, invalidValue());
    }

    /**
     * Поле содержит значение {@code null} или {@code false}.
     */
    public static Constraint<Boolean, Defect> notTrue() {
        return Constraint.fromPredicate(bool -> !bool, invalidValue());
    }

    public static <T> Constraint<List<T>, Defect> eachNotNull() {
        return fromPredicate(
                list -> list.stream().allMatch(Objects::nonNull), CollectionDefects.notContainNulls());
    }

    public static <T> Constraint<Set<T>, Defect> eachInSetNotNull() {
        return fromPredicate(
                list -> list.stream().allMatch(Objects::nonNull), CollectionDefects.notContainNulls());
    }

    public static Constraint<Set<Long>, Defect> eachInSetValidId() {
        return fromPredicate(
                set -> set.stream().allMatch(not(Predicates.lessThan(1L))), CommonDefects.validId());
    }

    public static <T> Constraint<Set<T>, Defect> eachInSet(Set<T> set) {
        return fromPredicate(set::containsAll, CollectionDefects.inCollection());
    }

    public static <T> Constraint<T, Defect> inSet(Set<T> set) {
        return fromPredicate(set::contains, CollectionDefects.inCollection());
    }

    public static <T> Constraint<T, Defect> inSet(Supplier<Set<T>> set) {
        // Supplier может быть медленным, а inSet может быть вызван из checkEach,
        // что повлечёт множественным вызовы Constraint#apply, поэтому используем memoize
        Supplier<Set<T>> lazySet = CommonUtils.memoize(set);
        return fromPredicate(
                item -> lazySet.get().contains(item),
                CollectionDefects.inCollection());
    }

    public static <T> Constraint<T, Defect> notInSet(Set<T> set) {
        return fromPredicate(not(set::contains), CollectionDefects.notInCollection());
    }

    public static Constraint<Long, Defect> validId() {
        return fromPredicate(not(Predicates.lessThan(1L)), CommonDefects.validId());
    }

    public static Constraint<Integer, Defect> validIntegerId() {
        return fromPredicate(not(Predicates.lessThan(1)), CommonDefects.validId());
    }

    public static <T> Constraint<Optional<T>, Defect> notEmpty() {
        return fromPredicate(Optional::isPresent, CommonDefects.requiredButEmpty());
    }
}
