package ru.yandex.direct.validation.constraint;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.utils.StringUtils.countDigits;


public class ContactsConstraints {
    private static final Pattern ICQ_PATTERN = Pattern.compile("^\\d{5,10}$");
    private static final Pattern PHONE_PATTERN = Pattern.compile("^\\+?[0-9\\-\\s()]+$");
    private static final Pattern TELEGRAM_PATTERN = Pattern.compile("^[0-9A-Za-z_]{5,32}$");
    private static final Pattern SKYPE_PATTERN = Pattern.compile("^[A-Za-z][0-9A-Za-z_\\-\\.\\:]{5,32}$");

    /**
     * Проверка формата введёного номера ICQ.
     * Номер должен содердать не менее пяти и не более десяти цифр и не содержать дефисов.
     */
    public static Constraint<String, Defect> validIcq() {
        return Constraint.fromPredicate(icq -> {
            // проверки на null и пустое значение делаются отдельно
            if (isBlank(icq)) {
                return true;
            }

            Matcher matcher = ICQ_PATTERN.matcher(icq);
            return matcher.matches();
        }, CommonDefects.invalidValue());
    }

    /**
     * Проверка формата введёного номера телефона.
     * Номер может быть в международной и местной нотации и содержать не менее пяти и не более пятнадцати цифр.
     */
    public static Constraint<String, Defect> validPhone() {
        return Constraint.fromPredicate(phone -> {
            // проверки на null и пустое значение делаются отдельно
            if (isBlank(phone)) {
                return true;
            }

            Matcher matcher = PHONE_PATTERN.matcher(phone);
            if (!matcher.matches()) {
                return false;
            }

            int countDigits = countDigits(phone);
            return 5 <= countDigits && countDigits <= 15;
        }, CommonDefects.invalidValue());
    }

    /**
     * Проверка имени (Username) пользователя из Телеграмма.
     * Имя должно содердать не менее пяти и не более тридцати двух символов (латинские буквы, цифры и нижнее подчёркивание) НЕ содеражть в начале знак @.
     */
    public static Constraint<String, Defect> validTelegram() {
        return Constraint.fromPredicate(telegram -> {
            // проверки на null и пустое значение делаются отдельно
            if (isBlank(telegram)) {
                return true;
            }

            Matcher matcher = TELEGRAM_PATTERN.matcher(telegram);
            return matcher.matches();
        }, CommonDefects.invalidValue());
    }

    /**
     * Проверка формата логина Skype.
     * Имя должно содердать не менее пяти и не более тридцати двух символов (латинские буквы, цифры, нижнее подчёркивание, дефис, точка).
     */
    public static Constraint<String, Defect> validSkype() {
        return Constraint.fromPredicate(skype -> {
            // проверки на null и пустое значение делаются отдельно
            if (isBlank(skype)) {
                return true;
            }

            Matcher matcher = SKYPE_PATTERN.matcher(skype);
            return matcher.matches();
        }, CommonDefects.invalidValue());
    }
}
