package ru.yandex.direct.validation.constraint;

import java.math.BigDecimal;

import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.NumberDefects;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public class NumberConstraints {

    private NumberConstraints() {
    }

    // numbers
    public static Constraint<Long, Defect> greaterThan(long min) {
        return fromPredicate(v -> v > min, NumberDefects.greaterThan(min));
    }

    public static Constraint<Integer, Defect> greaterThan(int min) {
        return fromPredicate(v -> v > min, NumberDefects.greaterThan(min));
    }

    public static Constraint<Double, Defect> greaterThan(double min) {
        return fromPredicate(v -> v > min, NumberDefects.greaterThan(min));
    }

    public static Constraint<BigDecimal, Defect> greaterThan(BigDecimal min) {
        return fromPredicate(v -> v.compareTo(min) > 0, NumberDefects.greaterThan(min));
    }

    public static Constraint<Long, Defect> lessThan(long max) {
        return fromPredicate(v -> v < max, NumberDefects.lessThan(max));
    }

    public static Constraint<Integer, Defect> lessThan(int max) {
        return fromPredicate(v -> v < max, NumberDefects.lessThan(max));
    }

    public static Constraint<Double, Defect> lessThan(double max) {
        return fromPredicate(v -> v < max, NumberDefects.lessThan(max));
    }

    public static Constraint<BigDecimal, Defect> lessThan(BigDecimal max) {
        return fromPredicate(v -> v.compareTo(max) < 0, NumberDefects.lessThan(max));
    }

    public static Constraint<BigDecimal, Defect> notGreaterThan(BigDecimal max) {
        return fromPredicate(v -> v.compareTo(max) <= 0, NumberDefects.lessThanOrEqualTo(max));
    }

    public static Constraint<BigDecimal, Defect> notGreaterThanOrEqualTo(BigDecimal max) {
        return fromPredicate(v -> v.compareTo(max) < 0, NumberDefects.lessThan(max));
    }

    public static Constraint<Long, Defect> notGreaterThan(long max) {
        return fromPredicate(v -> v <= max, NumberDefects.lessThanOrEqualTo(max));
    }

    public static Constraint<Double, Defect> notGreaterThan(double max) {
        return fromPredicate(v -> v <= max, NumberDefects.lessThanOrEqualTo(max));
    }

    public static Constraint<Integer, Defect> notGreaterThan(int max) {
        return fromPredicate(v -> v <= max, NumberDefects.lessThanOrEqualTo(max));
    }

    public static Constraint<BigDecimal, Defect> notLessThan(BigDecimal min) {
        return fromPredicate(v -> v.compareTo(min) >= 0, NumberDefects.greaterThanOrEqualTo(min));
    }

    public static Constraint<Long, Defect> notLessThan(long min) {
        return fromPredicate(v -> v >= min, NumberDefects.greaterThanOrEqualTo(min));
    }

    public static Constraint<Double, Defect> notLessThan(double min) {
        return fromPredicate(v -> v >= min, NumberDefects.greaterThanOrEqualTo(min));
    }

    public static Constraint<Integer, Defect> notLessThan(int min) {
        return fromPredicate(v -> v >= min, NumberDefects.greaterThanOrEqualTo(min));
    }

    public static Constraint<BigDecimal, Defect> notLessThanOrEqualTo(BigDecimal min) {
        return fromPredicate(v -> v.compareTo(min) > 0, NumberDefects.greaterThan(min));
    }

    public static Constraint<Integer, Defect> inRange(int min, int max) {
        return fromPredicate(Predicates.inRange(min, max), NumberDefects.inInterval(min, max));
    }

    public static Constraint<Long, Defect> inRange(long min, long max) {
        return fromPredicate(Predicates.inRange(min, max), NumberDefects.inInterval(min, max));
    }

    public static Constraint<BigDecimal, Defect> inRange(BigDecimal min, BigDecimal max) {
        return fromPredicate(Predicates.inRange(min, max), NumberDefects.inInterval(min, max));
    }
}
